package ru.yandex.direct.intapi.entity.clients;

import java.util.List;
import java.util.Objects;

import javax.ws.rs.GET;
import javax.ws.rs.POST;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.LoginOrUid;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.entity.clients.exception.PddLoginException;
import ru.yandex.direct.intapi.entity.clients.model.AgencyManagedUsersResponse;
import ru.yandex.direct.intapi.entity.clients.model.CheckClientStateRequest;
import ru.yandex.direct.intapi.entity.clients.model.CheckClientStateResponse;
import ru.yandex.direct.intapi.entity.clients.model.ClientIdResponse;
import ru.yandex.direct.intapi.entity.clients.model.ClientInfo;
import ru.yandex.direct.intapi.entity.clients.model.ClientState;
import ru.yandex.direct.intapi.entity.clients.model.ClientsAddOrGetRequest;
import ru.yandex.direct.intapi.entity.clients.model.ClientsAddOrGetResponse;
import ru.yandex.direct.intapi.entity.clients.model.ClientsGetBulkResponse;
import ru.yandex.direct.intapi.entity.clients.model.EnableFeatureRequest;
import ru.yandex.direct.intapi.entity.clients.model.OnClientCreateRequest;
import ru.yandex.direct.intapi.entity.clients.model.OperatorPermissions;
import ru.yandex.direct.intapi.entity.clients.service.IntapiClientService;
import ru.yandex.direct.tvm.AllowServices;
import ru.yandex.direct.web.core.exception.WebBadRequestException;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.model.WebSuccessResponse;

import static ru.yandex.direct.tvm.TvmService.CRM_PROD;
import static ru.yandex.direct.tvm.TvmService.CRM_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_AUTOTESTS;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_TEST;
import static ru.yandex.direct.tvm.TvmService.EDA_RESTAPP_MARKETING_PROD;
import static ru.yandex.direct.tvm.TvmService.EDA_RESTAPP_MARKETING_TEST;
import static ru.yandex.direct.tvm.TvmService.EDA_RESTAPP_PROD;
import static ru.yandex.direct.tvm.TvmService.EDA_RESTAPP_TEST;
import static ru.yandex.direct.tvm.TvmService.FREESERVICE;
import static ru.yandex.direct.tvm.TvmService.LPC_SWITCHCONSTRUCTOR_PROD;
import static ru.yandex.direct.tvm.TvmService.LPC_SWITCHCONSTRUCTOR_TEST;
import static ru.yandex.direct.tvm.TvmService.RMP_PROD;
import static ru.yandex.direct.tvm.TvmService.RMP_TEST;
import static ru.yandex.direct.tvm.TvmService.SMB;
import static ru.yandex.direct.tvm.TvmService.SPRAV_API_PROD;
import static ru.yandex.direct.tvm.TvmService.SPRAV_API_TEST;
import static ru.yandex.direct.tvm.TvmService.TYCOON_PROD;
import static ru.yandex.direct.tvm.TvmService.TYCOON_TEST;
import static ru.yandex.direct.tvm.TvmService.YDO_PROD;
import static ru.yandex.direct.tvm.TvmService.YDO_TEST;
import static ru.yandex.direct.tvm.TvmService.ZEN_PROD;
import static ru.yandex.direct.tvm.TvmService.ZEN_TEST;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@RestController
@RequestMapping(
        name = "clients",
        path = "/clients",
        produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Api(value = "API для получения данных по клиентам")
// не указываем @AllowNetworks специально - читающий метод может использовать кто угодно,
// intapi доступно только из интранета
public class ClientsController {
    private static final Logger logger = LoggerFactory.getLogger(ClientsController.class);

    private static final int PDD_CLIENT_ERROR_CODE = 1001;

    private IntapiClientService intapiClientService;

    @Autowired
    public ClientsController(IntapiClientService intapiClientService) {
        this.intapiClientService = intapiClientService;
    }

    @GET
    @ApiOperation(
            value = "получение информации по клиенту для AdFox",
            nickname = "getClientById",
            httpMethod = "GET"
    )
    @ApiResponses({
            @ApiResponse(code = 200, message = "список найденных клиентов", response = ClientsGetBulkResponse.class),
            @ApiResponse(code = 400, message = "ошибка запроса", response = ErrorResponse.class)})
    @RequestMapping(path = "get",
            method = RequestMethod.GET
    )
    public ClientsGetBulkResponse getClientInfoByIds(
            @ApiParam(value = "ID клиента", required = true)
            @RequestParam(value = "client_id") List<Long> clientIds,
            @RequestParam(value = "miss_money_fields", required = false) Boolean missMoneyFields) {
        return getClientInfoByIdsBulk(clientIds, missMoneyFields);
    }

    @POST
    @ApiOperation(
            value = "получение информации по клиентам для AdFox",
            nickname = "getClientByIds",
            httpMethod = "POST"
    )
    @ApiResponses({
            @ApiResponse(code = 200, message = "список найденных клиентов", response = ClientsGetBulkResponse.class),
            @ApiResponse(code = 400, message = "ошибка запроса", response = ErrorResponse.class)})
    @RequestMapping(path = "get",
            method = RequestMethod.POST
    )
    public ClientsGetBulkResponse getClientInfoByIdsBulk(
            @ApiParam(value = "список ID клиентов", required = true)
            @RequestBody List<Long> clientIds,
            @RequestParam(value = "miss_money_fields", required = false) Boolean missMoneyFields) {
        clientIds.removeIf(Objects::isNull);

        List<ClientInfo> clientInfos = intapiClientService.getClientInfoByIds(
                mapList(clientIds, ClientId::fromLong), missMoneyFields);

        return new ClientsGetBulkResponse().withClients(clientInfos);
    }

    @POST
    @ApiOperation(
            value = "Создает нового клиента либо возвращает уже имеющегося если он зарегистрирован в директе",
            nickname = "addOrGetClient",
            httpMethod = "POST"
    )
    @ApiResponses({
            @ApiResponse(code = 200, message = "clientId и uid клиента", response = ClientsAddOrGetResponse.class),
            @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
            @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class)})
    @RequestMapping(path = "addOrGet", method = RequestMethod.POST)
    @AllowServices(production = {TYCOON_PROD, YDO_PROD, SMB, FREESERVICE, EDA_RESTAPP_PROD, EDA_RESTAPP_MARKETING_PROD,
                    ZEN_PROD},
            testing = {TYCOON_TEST, DIRECT_DEVELOPER, DIRECT_AUTOTESTS, YDO_PROD, YDO_TEST, SMB, FREESERVICE,
                    EDA_RESTAPP_PROD, EDA_RESTAPP_TEST, EDA_RESTAPP_MARKETING_PROD, EDA_RESTAPP_MARKETING_TEST,
                    ZEN_TEST})
    public WebResponse addOrGetClient(
            @ApiParam(value = "запрос на добавление клиента", required = true)
            @RequestBody ClientsAddOrGetRequest request) {
        try {
            //По дефолту создаём со включенным api
            return intapiClientService.addOrGetClient(LoginOrUid.of(request.getLogin(), request.getUid()),
                    request.getFio(), request.getCountry(), request.getCurrency(), nvl(request.isWithApi(), true));
        } catch (PddLoginException e) {
            logger.error("Cannot create PDD client", e);
            return new WebErrorResponse(PDD_CLIENT_ERROR_CODE, e.getMessage());
        } catch (Exception e) {
            logger.error("Exception while creating client", e);
            return new WebErrorResponse(1000, e.getMessage());
        }
    }

    @POST
    @ApiOperation(
            value = "Возвращает статус клиента в директе и его API",
            nickname = "checkClientState",
            httpMethod = "POST"
    )
    @ApiResponses({
            @ApiResponse(code = 200, message = "ClientState клиента", response = CheckClientStateResponse.class)})
    @PostMapping(path = "checkClientState")
    @AllowServices(production = {TYCOON_PROD, YDO_PROD, SMB, FREESERVICE, EDA_RESTAPP_PROD,
            EDA_RESTAPP_MARKETING_PROD, RMP_PROD, ZEN_PROD},
            testing = {TYCOON_TEST, DIRECT_DEVELOPER, DIRECT_AUTOTESTS, YDO_PROD, YDO_TEST, SMB, FREESERVICE, RMP_TEST,
                    EDA_RESTAPP_PROD, EDA_RESTAPP_TEST, EDA_RESTAPP_MARKETING_PROD, EDA_RESTAPP_MARKETING_TEST,
                    ZEN_TEST})
    public CheckClientStateResponse checkClientState(
            @ApiParam(value = "запрос на определение статуса клиента в директе и его API", required = true)
            @RequestBody CheckClientStateRequest request) {
        return intapiClientService.checkClientState(LoginOrUid.of(request.getLogin(), request.getUid()), false,
                Boolean.TRUE.equals(request.getWithBalance()));
    }

    @GET
    @ApiOperation(
            value = "Возвращает статус клиента в Директе",
            nickname = "getClientState",
            httpMethod = "GET"
    )
    @ApiResponses({
            @ApiResponse(code = 200, message = "ClientState клиента", response = ClientState.class),
            @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
            @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class)})
    @GetMapping(path = "getClientState")
    @AllowServices(production = {DIRECT_DEVELOPER}, // TODO: вписать ещё TVM_ID заказчиков ручки
            testing = {DIRECT_DEVELOPER})
    public ClientState getClientState(
            @ApiParam(value = "паспортный uid пользователя")
            @RequestParam(value = "uid", required = false) Long uid,
            @ApiParam(value = "паспортный логин пользователя")
            @RequestParam(value = "login", required = false) String login) {
        if (login == null && uid == null) {
            throw new WebBadRequestException("Нужно указать логин или uid");
        }
        return intapiClientService.checkClientState(LoginOrUid.of(login, uid), true, false)
                .getClientState();
    }

    @GET
    @ApiOperation(
            value = "получение прав на клиента",
            nickname = "get_operator_permissions",
            httpMethod = "GET"
    )
    @ApiResponses({
            @ApiResponse(code = 400, message = "Не найден оператор или пользователь", response = ErrorResponse.class),
            @ApiResponse(code = 200, message = "Права на клиента", response = OperatorPermissions.class)
    })
    @RequestMapping(
            path = "get_operator_permissions",
            method = RequestMethod.GET
    )
    @AllowServices(
            production = {TYCOON_PROD, SPRAV_API_PROD, RMP_PROD, YDO_PROD},
            testing = {TYCOON_TEST, SPRAV_API_TEST, DIRECT_DEVELOPER, RMP_TEST, YDO_TEST}
    )
    public OperatorPermissions getOperatorPermissions(
            @ApiParam(value = "паспортный uid оператора")
            @RequestParam(value = "operator_uid") Long operatorUid,
            @ApiParam(value = "паспортный uid пользователя, к которому проверяется доступ")
            @RequestParam(value = "chief_uid") Long chiefUid
    ) {
        return intapiClientService.getOperatorPermissions(operatorUid, chiefUid);
    }

    @GET
    @ApiOperation(
            value = "Определение пользователей, к которым у агентства есть доступ",
            nickname = "getAgencyManagedUsers",
            httpMethod = "GET"
    )
    @ApiResponses({
            @ApiResponse(code = 400, message = "не найдено агентство", response = WebErrorResponse.class),
            @ApiResponse(
                    code = 200,
                    message = "список пользователей, к которым есть доступ",
                    response = AgencyManagedUsersResponse.class
            )}
    )
    @GetMapping(path = "getAgencyManagedUsers")
    @AllowServices(
            production = {YDO_PROD},
            testing = {YDO_TEST, DIRECT_DEVELOPER}
    )
    public WebResponse getAgencyManagedUsers(
            @ApiParam(value = "uid представителя агентства", required = true)
            @RequestParam(value = "agency_uid") Long agencyUid,
            @ApiParam(value = "список uid-ов пользователей, доступ к которым нужно проверить", required = true)
            @RequestParam(value = "user_ids") List<Long> userIds
    ) {
        return new AgencyManagedUsersResponse(intapiClientService.getAgencyManagedUsers(agencyUid, userIds));
    }

    @ApiOperation(
            value = "Получить идентификатор клиента по идентификатору пользователя",
            nickname = "getClientIdByUid",
            httpMethod = "GET"
    )
    @ApiResponses({
            @ApiResponse(code = 200, message = "идентификатор клиента", response = ClientIdResponse.class),
            @ApiResponse(code = 400, message = "ошибка запроса", response = WebErrorResponse.class)})
    @RequestMapping(path = "getClientIdByUid", method = RequestMethod.GET)
    @AllowServices(
            production = {LPC_SWITCHCONSTRUCTOR_PROD, CRM_PROD, RMP_PROD},
            testing = {LPC_SWITCHCONSTRUCTOR_TEST, CRM_TEST, RMP_TEST, DIRECT_DEVELOPER}
    )
    public WebResponse getClientIdByUid(
            @ApiParam(value = "идентификатор пользователя", required = true) @RequestParam(value = "uid") Long uid) {
        return intapiClientService.getClientIdByUid(uid);
    }

    @ApiOperation(
            value = "Включить фичу клиенту",
            nickname = "enableFeature",
            httpMethod = "POST")
    @RequestMapping(
            path = "enableFeature",
            method = RequestMethod.POST)
    @AllowServices(
            production = {DIRECT_WEB_PROD},
            testing = {DIRECT_WEB_TEST, DIRECT_DEVELOPER})
    public WebResponse enableFeature(
            @RequestBody EnableFeatureRequest request) {
        ClientId clientId = ClientId.fromLong(request.getClientId());
        FeatureName featureName = FeatureName.fromString(request.getFeatureName());

        return intapiClientService.enableFeature(clientId, featureName);
    }

    @POST
    @ApiOperation(
            value = "Выполнение действия сразу после создания клиента",
            nickname = "onClientCreate",
            httpMethod = "POST"
    )
    @ApiResponses({
            @ApiResponse(code = 400, message = "ошибка запроса", response = WebErrorResponse.class),
            @ApiResponse(code = 200, message = "статус обработки запроса", response = WebSuccessResponse.class
            )}
    )
    @RequestMapping(path = "onClientCreate",
            method = RequestMethod.POST)
    @AllowServices(
            production = {DIRECT_WEB_PROD},
            testing = {DIRECT_WEB_TEST, DIRECT_DEVELOPER}
    )
    public WebResponse onClientCreate(
            @RequestBody OnClientCreateRequest request) {
        ClientId clientId = ClientId.fromNullableLong(request.getClientId());

        return intapiClientService.onClientCreate(clientId);
    }
}
