package ru.yandex.direct.intapi.entity.clients.service;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import com.google.common.collect.Multimaps;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import ru.yandex.direct.communication.service.CommunicationEventService;
import ru.yandex.direct.core.entity.apifinancetokens.repository.ApiFinanceTokensRepository;
import ru.yandex.direct.core.entity.campaign.model.Wallet;
import ru.yandex.direct.core.entity.campaign.model.WalletRestMoney;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.campaign.service.WalletService;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.model.ClientNds;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.core.entity.client.service.AddClientOptions;
import ru.yandex.direct.core.entity.client.service.AddClientService;
import ru.yandex.direct.core.entity.client.service.ClientNdsService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.service.FeatureManagingService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.user.model.ApiEnabled;
import ru.yandex.direct.core.entity.user.model.ApiUser;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.ApiUserService;
import ru.yandex.direct.core.entity.user.service.BlackboxUserService;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.core.validation.defects.Defects;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Percent;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.LoginOrUid;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.model.UidClientIdLogin;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.IntApiException;
import ru.yandex.direct.intapi.entity.clients.exception.PddLoginException;
import ru.yandex.direct.intapi.entity.clients.model.CheckClientStateResponse;
import ru.yandex.direct.intapi.entity.clients.model.ClientIdResponse;
import ru.yandex.direct.intapi.entity.clients.model.ClientInfo;
import ru.yandex.direct.intapi.entity.clients.model.ClientRole;
import ru.yandex.direct.intapi.entity.clients.model.ClientState;
import ru.yandex.direct.intapi.entity.clients.model.ClientsAddOrGetResponse;
import ru.yandex.direct.intapi.entity.clients.model.OperatorPermissions;
import ru.yandex.direct.rbac.PpcRbac;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.utils.PassportUtils;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.model.WebSuccessResponse;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxCorrectResponse;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.isValidId;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.PassportUtils.normalizeLogin;

@Service
public class IntapiClientService {

    private static final Logger logger = LoggerFactory.getLogger(IntapiClientService.class);

    private final ClientService clientService;
    private final FeatureService featureService;
    private final AddClientService addClientService;
    private final UserService userService;
    private final ApiFinanceTokensRepository apiFinanceTokensRepository;
    private final ApiUserService apiUserService;
    private final FeatureManagingService featureManagingService;
    private final ShardHelper shardHelper;
    private final ClientRepository clientRepository;
    private final PpcRbac rbac;
    private final WalletService walletService;
    private final BlackboxUserService blackboxUserService;
    private final ClientNdsService clientNdsService;
    private final CampaignService campaignService;
    private final CommunicationEventService communicationEventService;

    @Autowired
    @SuppressWarnings("checkstyle:parameternumber")
    public IntapiClientService(ClientService clientService,
                               FeatureService featureService,
                               AddClientService addClientService,
                               UserService userService,
                               ApiFinanceTokensRepository apiFinanceTokensRepository,
                               ApiUserService apiUserService,
                               FeatureManagingService featureManagingService,
                               ShardHelper shardHelper,
                               ClientRepository clientRepository,
                               PpcRbac rbac,
                               WalletService walletService,
                               BlackboxUserService blackboxUserService,
                               ClientNdsService clientNdsService,
                               CampaignService campaignService,
                               CommunicationEventService communicationEventService) {
        this.clientService = clientService;
        this.featureService = featureService;
        this.addClientService = addClientService;
        this.userService = userService;
        this.apiFinanceTokensRepository = apiFinanceTokensRepository;
        this.apiUserService = apiUserService;
        this.featureManagingService = featureManagingService;
        this.shardHelper = shardHelper;
        this.clientRepository = clientRepository;
        this.rbac = rbac;
        this.walletService = walletService;
        this.blackboxUserService = blackboxUserService;
        this.clientNdsService = clientNdsService;
        this.campaignService = campaignService;
        this.communicationEventService = communicationEventService;
    }

    /**
     * По переданным {@code clientIds} возвращает информацию о найденных клиентах.
     * Если ничего не найдено, возвращается пустой список.
     */
    public List<ClientInfo> getClientInfoByIds(List<ClientId> clientIds, Boolean missMoneyFields) {
        // подчистим из входных данных null'ы
        clientIds.removeIf(Objects::isNull);
        if (clientIds.isEmpty()) {
            return emptyList();
        }

        Collection<Client> clients = clientService.massGetClient(clientIds);
        Map<ClientId, Boolean> cpmFeatureEnabled =
                featureService.isEnabledForClientIds(listToSet(clientIds), FeatureName.CPM_DEALS.getName());

        Map<ClientId, Currency> currencyByClientId = clientService.massGetWorkCurrency(clientIds);
        if (missMoneyFields == null) {
            missMoneyFields = false;
        }
        Map<ClientId, BigDecimal> balanceByClientId = missMoneyFields
                ? Collections.emptyMap()
                : getBalanceByClientId(clientIds, currencyByClientId);
        Map<ClientId, Percent> clientNdsByClientId = missMoneyFields
                ? Collections.emptyMap()
                : clientNdsService
                .massGetEffectiveClientNdsByIds(clientIds).stream()
                .collect(toMap(c -> ClientId.fromLong(c.getClientId()), ClientNds::getNds));

        return clients.stream()
                .map(client -> toClientInfo(
                        client, cpmFeatureEnabled, currencyByClientId, balanceByClientId, clientNdsByClientId))
                .collect(Collectors.toList());
    }

    /**
     * По переданным {@code walletsByClientId} и {@code currencyByClientId} возвращает информацию о балансе клиентов.
     */
    private Map<ClientId, BigDecimal> getBalanceByClientId(
            Collection<ClientId> clientIds,
            Map<ClientId, Currency> currencyByClientId
    ) {
        Map<ClientId, Collection<Wallet>> walletsByClientId = Multimaps.index(
                walletService.massGetWallets(clientIds), Wallet::getClientId).asMap();

        // кошельки по их идентификатору, он же campaignId (cid)
        Map<Long, Wallet> walletById = EntryStream.of(walletsByClientId)
                .mapKeys(currencyByClientId::get)
                // фильтруем кошельки по валюте клиента, должно остаться не больше одного
                .mapKeyValue((clientCurrency, wallets) -> StreamEx.of(wallets)
                        .findFirst(w -> w.getCampaignsCurrency().equals(clientCurrency))
                        .orElse(null))
                .nonNull()
                .mapToEntry(Wallet::getWalletCampaignId, Function.identity())
                .toMap();

        List<WalletRestMoney> restMoney = shardHelper.groupByShard(walletById.keySet(), ShardKey.CID).stream()
                .mapKeyValue(campaignService::getWalletsRestMoneyByWalletCampaignIds)
                .toFlatList(Map::values);

        return StreamEx.of(restMoney)
                .mapToEntry(WalletRestMoney::getWalletId, WalletRestMoney::getRest)
                .mapKeys(walletById::get)
                .mapKeys(Wallet::getClientId)
                .mapValues(money -> money.roundToCentDown().bigDecimalValue())
                .toMap();
    }

    /**
     * Возвращает информацию о клиенте
     */
    private ClientInfo toClientInfo(
            Client client,
            Map<ClientId, Boolean> cpmFeatureEnabled,
            Map<ClientId, Currency> currencyByClientId,
            Map<ClientId, BigDecimal> balanceByClientId,
            Map<ClientId, Percent> clientNdsByClientId
    ) {
        ClientId clientId = ClientId.fromLong(client.getClientId());
        Currency currency = currencyByClientId.get(clientId);
        Percent clientNds = clientNdsByClientId.get(clientId);

        return new ClientInfo()
                .withClientId(client.getId())
                .withName(client.getName())
                // null тут не должен появиться, но если появится, лучше отдать строчку "null"
                .withRole(String.valueOf(client.getRole()))
                .withCurrencyCode(ifNotNull(currency, Currency::getCode))
                .withBalance(balanceByClientId.get(clientId))
                .withNds(ifNotNull(clientNds, Percent::asPercent))
                .withCountryRegionId(client.getCountryRegionId())
                .withHasContract((client.getRole().equals(RbacRole.AGENCY)) ?
                        Optional.ofNullable(cpmFeatureEnabled.get(ClientId.fromLong(client.getId())))
                                .orElse(false) : null);
    }

    /**
     * Создает нового клиента и возвращает существующего на основании логина.
     */
    public WebResponse addOrGetClient(@Nonnull LoginOrUid loginOrUid,
                                      String fio, long country, CurrencyCode currency, boolean enableApi) {
        UidClientIdLogin uidClientIdLogin = addOrGetClient(loginOrUid, fio, currency, country, enableApi);

        insertPremoderationFeature(uidClientIdLogin.getClientId());

        Long uid = uidClientIdLogin.getUid();
        if (enableApi) {
            apiUserService.allowFinancialOperationsAcceptApiOffer(uid);
        }
        long clientId = uidClientIdLogin.getClientId().asLong();
        String financeToken = enableApi ? apiFinanceTokensRepository.getMasterToken(uid)
                .orElseGet(() -> apiFinanceTokensRepository.createAndGetMasterToken(uid)) : null;
        String login = uidClientIdLogin.getLogin();
        return new ClientsAddOrGetResponse(clientId, uid, financeToken, login);
    }

    /**
     * Получить клиента по {@code loginOrUid} или создать клиента, если запрошенный отсутствует.
     * Если клиента не удалось найти в БД по логину, то в Blackbox будет запрошен uid пользователя,
     * чтобы попытаться найти его по uid-у, т.к. переданный логин может быть алиасом (не основным логином).
     */
    public UidClientIdLogin addOrGetClient(
            @Nonnull LoginOrUid loginOrUid,
            String fio,
            CurrencyCode currency,
            long country,
            boolean enableApi
    ) {
        User user = loginOrUid.map(login -> userService.getUserByLogin(normalizeLogin(login)), userService::getUser);
        if (user != null) {
            return updateExistingClient(user, enableApi);
        }
        // То, что пользователя нет в нашей базе по указанному логину, еще не означает, что у нас нет этого пользователя
        // Мог быть указан логин == телефонный алиас. Поэтому нужно сделать запрос в blackbox за uid-ом такого
        // пользователя, и попробовать найти в нашей базе его по uid-у
        Result<BlackboxCorrectResponse> blackboxResult = blackboxUserService.getAndCheckUser(loginOrUid);
        if (!blackboxResult.isSuccessful()) {
            throw buildCreateClientError(Result.broken(blackboxResult.getValidationResult()));
        }
        long uid = blackboxResult.getResult().getUid().get().getUid();
        var login = PassportUtils.normalizeLogin(blackboxResult.getResult().getLogin().get());
        User blackboxUidUser = userService.getUser(uid);

        // Клиент нашелся по uid-у
        if (blackboxUidUser != null) {
            logger.info("Can't find a db user by the requested login but found by the login {}", login);
            return updateExistingClient(blackboxUidUser, enableApi);
        }
        // Клиента все-таки нет, поэтому нужно его создать
        var newUidClientId = createNewClient(blackboxResult, loginOrUid, fio, country, currency, enableApi);

        return UidClientIdLogin.of(newUidClientId, login);
    }

    /**
     * Создает нового клиента. Бросает {@link RuntimeException} в случае ошибки.
     */
    private UidAndClientId createNewClient(Result<BlackboxCorrectResponse> blackboxResult,
                                           LoginOrUid loginOrUid, String fio,
                                           long country, CurrencyCode currency,
                                           boolean enableApi) throws RuntimeException {
        AddClientOptions addClientOptions = AddClientOptions.defaultOptions()
                .withPaymentBeforeModerationAllowed(true)
                .withUseExistingIfRequired(true)
                .withDisableApi(!enableApi)
                .withWallet(true);
        Result<UidAndClientId> serviceResult = addClientService
                .processRequest(blackboxResult, loginOrUid, fio, country, currency, RbacRole.CLIENT, addClientOptions);
        if (serviceResult.isSuccessful()) {
            return serviceResult.getResult();
        }
        throw buildCreateClientError(serviceResult);
    }

    private RuntimeException buildCreateClientError(Result<UidAndClientId> serviceResult) {
        DefectInfo<Defect> pddDefectInfo = serviceResult.getErrors().stream()
                .filter(di -> di.getDefect().equals(Defects.pddLogin()))
                .findFirst().orElse(null);
        if (pddDefectInfo != null) {
            return new PddLoginException(pddDefectInfo.toString());
        }
        return new RuntimeException(serviceResult.getErrors().get(0).toString());
    }

    /**
     * Обновляет существующего клиента нужным образом
     */
    private UidClientIdLogin updateExistingClient(User user, boolean enableApi) {
        if (enableApi) {
            clientRepository.insertDefaultClientsApiOptions(
                    shardHelper.getShardByClientId(user.getClientId()), user.getClientId(), false);
        }
        return UidClientIdLogin.of(user.getUid(), user.getClientId(), user.getLogin());
    }

    /**
     * Добавляем клиенту фичу payment_before_moderation для оплаты кампаний до модерации
     */
    private void insertPremoderationFeature(ClientId clientId) {
        featureManagingService.enableFeatureForClient(clientId, FeatureName.PAYMENT_BEFORE_MODERATION);
    }


    /**
     * Поведение совпадает c {@link #checkClientState(LoginOrUid, boolean, boolean)}, где второй параметр {@code null}
     *
     * @param loginOrUid логин или uid
     * @return состояние клиента {@link CheckClientStateResponse}
     */
    public CheckClientStateResponse checkClientState(@Nonnull LoginOrUid loginOrUid) {
        return checkClientState(loginOrUid, false, false);
    }

    /**
     * Возвращает состояние клиента {@link CheckClientStateResponse}.
     * Если указать {@code stateOnly} {@code true}, то в ответе точно заполнено будет только ClientState.
     * Если указать {@code stateOnly} {@code false} и {@code withBalance} {@code true},
     * то в ответе будет поле {@code balance}.
     * В противном случае его не будет.
     *
     * @param loginOrUid  — логин или uid
     * @param stateOnly   — вычислить только ClientState
     * @param withBalance — нужно ли в ответе поле balance, несовместимо со включенным stateOnly
     * @return состояние клиента {@link CheckClientStateResponse}
     */
    public CheckClientStateResponse checkClientState(@Nonnull LoginOrUid loginOrUid,
                                                     boolean stateOnly,
                                                     boolean withBalance) {
        ApiUser user = loginOrUid.map(login -> apiUserService.getUser(normalizeLogin(login)), apiUserService::getUser);
        if (user == null) {
            // проверяем, что клиента можно создать в Директе
            Result<BlackboxCorrectResponse> blackboxResult = blackboxUserService.getAndCheckUser(loginOrUid);
            if (blackboxResult.isSuccessful()) {
                return new CheckClientStateResponse(ClientState.NOT_EXISTS, null);
            } else {
                return new CheckClientStateResponse(ClientState.CAN_NOT_BE_CREATED,
                        blackboxResult.getErrors().get(0).getDefect().defectId().getCode());
            }
        }

        ClientState clientState;
        if (user.getStatusBlocked()) {
            clientState = ClientState.BLOCKED;
        } else {
            clientState = calculateDirectNonBlockedClientState(user);
        }

        if (stateOnly) {
            return new CheckClientStateResponse(clientState, null);
        }

        // для консистентности с {@link DirectCookieAuthProvider::resolveSubjectUser}
        String loginInDirect = shardHelper.getLoginByUid(user.getUid());

        BigDecimal balance = null;
        CurrencyCode currencyCode = null;
        if (withBalance) {
            var clientId = user.getClientId();
            var currency = clientService.getWorkCurrency(clientId);
            balance = getBalanceByClientId(List.of(clientId), Map.of(clientId, currency)).get(clientId);
            currencyCode = currency.getCode();
        }

        return new CheckClientStateResponse(clientState, loginInDirect, getClientRole(user),
                walletService.hasEnabledSharedWallet(user), balance, currencyCode);
    }

    /**
     * Вычисляет ClientState для клиентов, уже заведённых в директе, и незаблокированных напрямую в интерфейсе директа
     */
    @Nonnull
    private ClientState calculateDirectNonBlockedClientState(@Nonnull ApiUser user) {
        if (user.getApiEnabled() == ApiEnabled.NO) {
            return ClientState.API_BLOCKED;
        }
        return Boolean.TRUE.equals(user.getApiOfferAccepted()) ? ClientState.API_ENABLED : ClientState.API_DISABLED;
    }

    @Nonnull
    private ClientRole getClientRole(@Nonnull User user) {
        if (user.getRole() == RbacRole.AGENCY) {
            return ClientRole.AGENCY;
        } else if (user.getRole() == RbacRole.CLIENT) {
            if (user.getAgencyClientId() == null) {
                return ClientRole.CLIENT;
            }
            if (clientService.isSuperSubclient(user.getClientId())) {
                return ClientRole.AGENCY_SUBCLIENT;
            }
            return ClientRole.AGENCY_READ_ONLY_SUBCLIENT;
        } else {
            return ClientRole.OTHER;
        }
    }

    /**
     * Возвращает разрешенные действия {@link OperatorPermissions} оператора к клиенту.
     *
     * @param operatorUid uid оператора
     * @param chiefUid    uid шефа
     * @return разрешенные действия {@link OperatorPermissions}
     */
    public OperatorPermissions getOperatorPermissions(Long operatorUid, Long chiefUid) {
        validateAuthParams(operatorUid, chiefUid);

        return massGetOperatorPermissions(operatorUid, List.of(chiefUid)).getOrDefault(
                chiefUid, new OperatorPermissions.Builder().build());
    }

    /**
     * Получить идентификатор клиента в формате {@link ClientIdResponse} или {@link WebErrorResponse},
     * если клиент не найден.
     *
     * @param uid идентификатор пользователя
     * @return идентификатор клиента
     */
    public WebResponse getClientIdByUid(Long uid) {
        try {
            Long clientId = shardHelper.getClientIdByUid(uid);
            return new ClientIdResponse(clientId);
        } catch (IllegalArgumentException e) {
            return new WebErrorResponse(HttpStatus.BAD_REQUEST.value(), e.getMessage());
        }
    }


    /**
     * Получить разрешенные действия {@link OperatorPermissions} оператора к каждому переданному пользователю
     *
     * @param operatorUid идентификатор оператора
     * @param userIds     список проверяемых пользователей
     * @return права оператора применительно к каждому пользователю
     */
    public Map<Long, OperatorPermissions> massGetOperatorPermissions(Long operatorUid, List<Long> userIds) {
        return EntryStream.of(rbac.getAccessTypes(operatorUid, userIds)).mapValues(type -> {
            var builder = new OperatorPermissions.Builder();
            OperatorPermissions.Permission.ofRbacAccessType(type).forEach(builder::addPermission);
            return builder.build();
        }).toMap();
    }

    /**
     * Оставить из списка переданных пользователей
     * только тех, на кого у данного представителя агентства есть права
     *
     * @param agencyUid uid представителя агентства
     * @param userIds   список пользователей
     * @return список пользователей, на которых у агентства есть права
     */
    public List<Long> getAgencyManagedUsers(Long agencyUid, List<Long> userIds) {

        try {
            Assert.isTrue(isValidId(agencyUid), String.format("incorrect uid %s", agencyUid));
            Long clientId = shardHelper.getClientIdByUid(agencyUid);
            Assert.notNull(clientId, String.format("uid %s not found", agencyUid));
            var client = clientService.getClient(ClientId.fromLong(clientId));
            if (client == null || client.getRole() != RbacRole.AGENCY) {
                return List.of();
            }
        } catch (IllegalArgumentException e) {
            throw new IntApiException(HttpStatus.BAD_REQUEST,
                    new ErrorResponse(ErrorResponse.ErrorCode.BAD_PARAM, e.getLocalizedMessage()));
        }

        Map<Long, OperatorPermissions> result = massGetOperatorPermissions(agencyUid, userIds);

        return result.entrySet().stream()
                .filter(e -> !e.getValue().getPermissions().isEmpty())
                .map(Map.Entry::getKey)
                .collect(Collectors.toList());
    }

    /**
     * Проверяет, что пользователи с заданными uid'ами существуют.
     *
     * @param operatorUid uid оператора
     * @param chiefUid    uid шефа
     * @throws IntApiException если uid оператора или шефа невалидный
     */
    private void validateAuthParams(Long operatorUid, Long chiefUid) throws IntApiException {
        try {
            Assert.isTrue(isValidId(operatorUid), "incorrect param operator_uid");
            Assert.isTrue(isValidId(chiefUid), "incorrect param chief_uid");
            Map<Long, Long> clientIdByUid = shardHelper.getClientIdsByUids(Arrays.asList(operatorUid, chiefUid));
            Assert.notNull(clientIdByUid.get(operatorUid), String.format("operator_uid %s not found", operatorUid));
            Assert.notNull(clientIdByUid.get(chiefUid), String.format("chief_uid %s not found", chiefUid));
        } catch (IllegalArgumentException e) {
            throw new IntApiException(HttpStatus.BAD_REQUEST,
                    new ErrorResponse(ErrorResponse.ErrorCode.BAD_PARAM, e.getLocalizedMessage()));
        }
    }

    public WebResponse enableFeature(ClientId clientId, FeatureName featureName) {
        featureManagingService.enableFeatureForClient(clientId, featureName);

        return new WebSuccessResponse();
    }

    /**
     * Выполняет определенную последовательность операций сразу создания клиента
     *
     * @param clientId идентификатор клиента
     * @return WebSuccessResponse если всё выполнено успешно
     */
    public WebResponse onClientCreate(ClientId clientId) {
        try {
            Assert.notNull(clientId, "empty clientid");
            Assert.isTrue(isValidId(clientId.asLong()), String.format("incorrect clientid format %s",
                    clientId.asLong()));

            communicationEventService.processOnClientCreateEvents(clientId);
        } catch (IllegalArgumentException e) {
            throw new IntApiException(HttpStatus.BAD_REQUEST,
                    new ErrorResponse(ErrorResponse.ErrorCode.BAD_PARAM, e.getLocalizedMessage()));
        } catch (RuntimeException ex) {
            throw new IntApiException(HttpStatus.INTERNAL_SERVER_ERROR,
                    new ErrorResponse(ErrorResponse.ErrorCode.INTERNAL_ERROR, ex.getLocalizedMessage()));
        }

        return new WebSuccessResponse();
    }
}
