package ru.yandex.direct.intapi.entity.connect.controller;

import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.core.MediaType;

import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.entity.connect.model.AddRoleRequest;
import ru.yandex.direct.intapi.entity.connect.model.IdmResponse;
import ru.yandex.direct.intapi.entity.connect.model.InfoResponse;
import ru.yandex.direct.intapi.entity.connect.model.RemoveRoleRequest;
import ru.yandex.direct.intapi.entity.connect.model.RequestFields;
import ru.yandex.direct.intapi.entity.connect.model.SubjectType;
import ru.yandex.direct.intapi.entity.connect.service.ConnectIdmAddRoleService;
import ru.yandex.direct.intapi.entity.connect.service.ConnectIdmGetPagedRolesService;
import ru.yandex.direct.intapi.entity.connect.service.ConnectIdmInfoService;
import ru.yandex.direct.intapi.entity.connect.service.ConnectIdmRemoveRoleService;
import ru.yandex.direct.tvm.AllowServices;

import static ru.yandex.direct.intapi.entity.connect.model.ConnectIdmConstants.RESOURCE_ID;
import static ru.yandex.direct.intapi.entity.connect.service.ConnectIdmGetPagedRolesService.CLIENTS_LIMIT;
import static ru.yandex.direct.intapi.entity.connect.service.ConnectIdmGetPagedRolesService.LAST_CLIENT_ID;
import static ru.yandex.direct.intapi.entity.connect.service.ConnectIdmGetPagedRolesService.LAST_SHARD;
import static ru.yandex.direct.tvm.TvmService.DIRECTORY_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.IDM_B2B_PROD;
import static ru.yandex.direct.tvm.TvmService.IDM_B2B_TEST;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.JsonUtils.fromJson;

@RestController
@RequestMapping(ConnectIdmRolesController.CONTROLLER_PATH)
@AllowServices(production = {IDM_B2B_PROD, DIRECTORY_PROD},
        testing = {IDM_B2B_TEST, DIRECTORY_PROD /*да, продакшен*/, DIRECT_DEVELOPER})
public class ConnectIdmRolesController {
    //Параметры
    private static final String ID = "id";
    private static final String PATH = "path";
    private static final String FIELDS = "fields";
    private static final String ORG_ID = "org_id";
    private static final String SUBJECT_TYPE = "subject_type";

    static final String CONTROLLER_PATH = "connect/idm";
    private static final String INFO_PATH = "info";
    private static final String GET_ROLES_PATH = "get-roles";
    private static final String ADD_ROLE_PATH = "add-role";
    private static final String REMOVE_ROLE_PATH = "remove-role";
    public static final String GET_ROLES_BASE_PATH = String.format("/%s/%s", CONTROLLER_PATH, GET_ROLES_PATH);


    private final ConnectIdmInfoService connectIdmInfoService;
    private final ConnectIdmGetPagedRolesService connectIdmGetPagedRolesService;
    private final ConnectIdmAddRoleService connectIdmAddRoleService;
    private final ConnectIdmRemoveRoleService connectIdmRemoveRoleService;


    @Autowired
    public ConnectIdmRolesController(
            ConnectIdmInfoService connectIdmInfoService,
            ConnectIdmGetPagedRolesService connectIdmGetPagedRolesService,
            ConnectIdmAddRoleService connectIdmAddRoleService,
            ConnectIdmRemoveRoleService connectIdmRemoveRoleService

    ) {
        this.connectIdmInfoService = connectIdmInfoService;
        this.connectIdmGetPagedRolesService = connectIdmGetPagedRolesService;
        this.connectIdmAddRoleService = connectIdmAddRoleService;
        this.connectIdmRemoveRoleService = connectIdmRemoveRoleService;
    }

    /**
     * Выдает статический json следующего вида:
     * {
     * "roles": {
     * "slug": "direct",
     * "name": "сервис",
     * "values": {
     * "organization": {
     * "name": "организация",
     * "roles": {
     * "values": {
     * "associated": "привязанная"
     * },
     * "slug": "role",
     * "name": "роль"
     * }
     * },
     * "user": {
     * "name": "пользователь",
     * "roles": {
     * "values": {
     * "chief": "основной представитель",
     * "employee": "представитель"
     * },
     * "slug": "role",
     * "name": "роль"
     * }
     * }
     * }
     * },
     * "fields": [
     * {
     * "slug": "resource_id",
     * "name": "идентификатор клиента в Директе",
     * "required": true,
     * "type": "charfield"
     * }
     * ],
     * "code": 0
     * }
     **/
    @GET
    @ApiOperation(
            value = "получить описание ролей Директа",
            nickname = INFO_PATH,
            httpMethod = "GET"
    )
    @ApiResponses({
            @ApiResponse(code = 403, message = "Недостаточно прав для выполнения операции",
                    response = ErrorResponse.class),
            @ApiResponse(code = 400, message = "Некорректный запрос",
                    response = ErrorResponse.class)
    })
    @RequestMapping(path = INFO_PATH,
            method = RequestMethod.GET,
            produces = MediaType.APPLICATION_JSON
    )
    public InfoResponse getInfo() {
        return connectIdmInfoService.getInfo();
    }

    @GET
    @ApiOperation(
            value = "получить логины и роли директа",
            nickname = GET_ROLES_PATH,
            httpMethod = "GET"
    )
    @ApiResponses({
            @ApiResponse(code = 403, message = "Недостаточно прав для выполнения операции",
                    response = ErrorResponse.class),
            @ApiResponse(code = 400, message = "Некорректный запрос",
                    response = ErrorResponse.class)
    })
    @RequestMapping(path = GET_ROLES_PATH,
            method = RequestMethod.GET,
            produces = MediaType.APPLICATION_JSON
    )
    public IdmResponse getRoles(
            @RequestParam(name = RESOURCE_ID, required = false) Long resourceId,
            @RequestParam(name = LAST_SHARD, required = false) Integer lastShard,
            @RequestParam(name = LAST_CLIENT_ID, required = false) Long lastClientId,
            @RequestParam(name = CLIENTS_LIMIT, required = false) Integer clientsLimit) {
        if (resourceId != null) {
            return connectIdmGetPagedRolesService.getSingleClientRoles(ClientId.fromLong(resourceId));
        }
        return connectIdmGetPagedRolesService.getRolesPage(ClientId.fromNullableLong(lastClientId), lastShard,
                clientsLimit);
    }

    @POST
    @ApiOperation(
            value = "выдать роль",
            nickname = ADD_ROLE_PATH,
            httpMethod = "POST"
    )
    @ApiResponses({
            @ApiResponse(code = 403, message = "Недостаточно прав для выполнения операции",
                    response = ErrorResponse.class),
            @ApiResponse(code = 400, message = "Некорректный запрос",
                    response = ErrorResponse.class)
    })
    @RequestMapping(path = ADD_ROLE_PATH,
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_FORM_URLENCODED,
            produces = MediaType.APPLICATION_JSON
    )
    public IdmResponse addRole(
            @RequestParam(name = ID, required = false) Long id,
            @RequestParam(name = PATH, required = false) String path,
            @RequestParam(name = FIELDS, required = false) String fieldsString,
            @RequestParam(name = ORG_ID, required = false) Long orgId,
            @RequestParam(name = SUBJECT_TYPE, required = false) String subjectTypeString) {

        AddRoleRequest requestInfo = new AddRoleRequest()
                .setId(id)
                .setPath(path)
                .setFields(requestFieldsFromString(fieldsString))
                .setOrgId(orgId)
                .setSubjectType(subjectTypeFromString(subjectTypeString));

        return connectIdmAddRoleService.addRole(requestInfo);
    }

    @POST
    @ApiOperation(
            value = "отозвать роль",
            nickname = REMOVE_ROLE_PATH,
            httpMethod = "POST"
    )
    @ApiResponses({
            @ApiResponse(code = 403, message = "Недостаточно прав для выполнения операции",
                    response = ErrorResponse.class),
            @ApiResponse(code = 400, message = "Некорректный запрос",
                    response = ErrorResponse.class)
    })
    @RequestMapping(path = REMOVE_ROLE_PATH,
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_FORM_URLENCODED,
            produces = MediaType.APPLICATION_JSON
    )
    public IdmResponse removeRole(
            @RequestParam(name = ID, required = false) Long id,
            @RequestParam(name = PATH, required = false) String path,
            @RequestParam(name = FIELDS, required = false) String fieldsString,
            @RequestParam(name = ORG_ID, required = false) Long orgId,
            @RequestParam(name = SUBJECT_TYPE, required = false) String subjectTypeString) {
        RemoveRoleRequest requestInfo = new RemoveRoleRequest()
                .setId(id)
                .setPath(path)
                .setFields(requestFieldsFromString(fieldsString))
                .setOrgId(orgId)
                .setSubjectType(subjectTypeFromString(subjectTypeString));

        return connectIdmRemoveRoleService.removeRole(requestInfo);
    }

    private SubjectType subjectTypeFromString(@RequestParam("subject_type") String subjectTypeString) {
        return ifNotNull(subjectTypeString, SubjectType::fromString);
    }

    private RequestFields requestFieldsFromString(@RequestParam("fields") String fieldsString) {
        return ifNotNull(fieldsString, l -> fromJson(l, RequestFields.class));
    }

}
