package ru.yandex.direct.intapi.entity.connect.service;

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientChiefService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.AddUserService;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.core.service.integration.balance.BalanceService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.intapi.entity.connect.model.AddPersonalRoleRequest;
import ru.yandex.direct.intapi.entity.connect.model.AddRoleRequest;
import ru.yandex.direct.intapi.entity.connect.model.ConnectError;
import ru.yandex.direct.intapi.entity.connect.model.ConnectIdmRoles;
import ru.yandex.direct.intapi.entity.connect.model.IdmErrorResponse;
import ru.yandex.direct.intapi.entity.connect.model.IdmResponse;
import ru.yandex.direct.intapi.entity.connect.model.IdmSuccessResponse;
import ru.yandex.direct.intapi.entity.connect.model.SubjectType;
import ru.yandex.direct.intapi.validation.kernel.TranslatableIntapiDefect;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.presentation.DefectPresentationRegistry;

import static org.apache.commons.lang.StringUtils.isBlank;
import static org.slf4j.LoggerFactory.getLogger;
import static ru.yandex.direct.intapi.common.converter.DefectsConverter.defectsToString;
import static ru.yandex.direct.intapi.entity.connect.model.DefectConvertor.mapDefectsToErrorCode;

@Service
@ParametersAreNonnullByDefault
public class ConnectIdmAddRoleService {
    private static final Logger logger = getLogger(ConnectIdmAddRoleService.class);

    private final ClientService clientService;
    private final UserService userService;
    private final BalanceService balanceService;
    private final FreelancerService freelancerService;
    private final AddUserService addUserService;
    private final ClientChiefService clientChiefService;
    private final DefectPresentationRegistry<TranslatableIntapiDefect> defectPresentationRegistry;
    private final TranslationService translationService;


    @SuppressWarnings("checkstyle:parameternumber")
    public ConnectIdmAddRoleService(ClientService clientService,
                                    UserService userService,
                                    BalanceService balanceService,
                                    FreelancerService freelancerService,
                                    AddUserService addUserService,
                                    ClientChiefService clientChiefService,
                                    DefectPresentationRegistry<TranslatableIntapiDefect> defectPresentationRegistry,
                                    TranslationService translationService) {
        this.clientService = clientService;
        this.userService = userService;
        this.balanceService = balanceService;
        this.freelancerService = freelancerService;
        this.addUserService = addUserService;
        this.clientChiefService = clientChiefService;
        this.defectPresentationRegistry = defectPresentationRegistry;
        this.translationService = translationService;
    }

    public IdmResponse addRole(AddRoleRequest addRoleRequest) {
        Optional<ConnectError> error = validate(addRoleRequest);
        if (error.isPresent()) {
            return new IdmErrorResponse(error.get());
        }

        ConnectIdmRoles role = ConnectIdmRoles.getRoleByPath(addRoleRequest.getPath());
        if (role == ConnectIdmRoles.ASSOCIATED) {
            return addOrgRole(addRoleRequest);
        }
        return addPersonalRole(addRoleRequest);
    }

    private IdmResponse addOrgRole(AddRoleRequest request) {
        ClientId clientId = extractClientId(request);
        Client client = clientService.getClient(clientId);
        if (client == null) {
            return new IdmErrorResponse(ConnectError.RESOURCE_NOT_FOUND);
        }
        if (client.getRole() != RbacRole.CLIENT) {
            return new IdmErrorResponse(ConnectError.RESOURCE_HAS_NO_CLIENT_ROLE);
        }
        boolean isFreelancer = freelancerService.isFreelancer(clientId);
        if (isFreelancer) {
            return new IdmErrorResponse(ConnectError.RESOURCE_CANNOT_BE_FREELANCER);
        }
        Long reqOrgId = request.getId();
        if (reqOrgId.equals(client.getConnectOrgId())) {
            return new IdmSuccessResponse();
        }
        List<Client> clientsByConnectOrgId =
                clientService.getClientsByConnectOrgId(Collections.singletonList(reqOrgId));
        if (!clientsByConnectOrgId.isEmpty()) {
            return new IdmErrorResponse(ConnectError.ID_ASSOCIATED_TO_ANOTHER_RESOURCE);
        }
        clientService.updateClientConnectOrgId(clientId.asLong(), reqOrgId);
        return new IdmSuccessResponse();
    }

    private IdmResponse addPersonalRole(AddRoleRequest addRoleRequest) {
        AddPersonalRoleRequest request = parseAddRoleRequest(addRoleRequest);
        ConnectError error = validatePersonalRoleRequest(request);
        if (error != null) {
            return new IdmErrorResponse(error);
        }
        return addPersonalRole(request);
    }

    private AddPersonalRoleRequest parseAddRoleRequest(AddRoleRequest request) {
        return new AddPersonalRoleRequest()
                .setOrgId(request.getOrgId())
                .setClientId(ClientId.fromLong(request.getFields().getResourceId()))
                .setUserId(request.getId())
                .setRole(ConnectIdmRoles.getRoleByPath(request.getPath()));
    }


    private ConnectError validatePersonalRoleRequest(AddPersonalRoleRequest request) {
        Client client = clientService.getClient(request.getClientId());
        if (client == null) {
            // клиент не привязан к организации
            return ConnectError.RESOURCE_NOT_FOUND;
        }
        if (!Objects.equals(client.getConnectOrgId(), request.getOrgId())) {
            // организация не совпадает
            return ConnectError.RESOURCE_NOT_ASSOCIATED;
        }
        //TODO : это костыль, нужно протащить в BalanceService нормальный метод, для проверки существования клиента.
        Optional<ClientId> clientIdByUid = balanceService.findClientIdByUid(client.getChiefUid());
        if (clientIdByUid.isEmpty()) {
            // клиент не привязан к организации
            return ConnectError.CLIENT_NOT_FOUND_IN_BALANCE;
        }

        // userId не привязан к другому клиенту
        User user = userService.getUser(request.getUserId());
        if (user != null && !Objects.equals(user.getClientId().asLong(), client.getClientId())) {
            return ConnectError.USER_ASSOCIATED_TO_ANOTHER_RESOURCE;
        }
        return null;
    }

    private IdmResponse addPersonalRole(AddPersonalRoleRequest request) {
        Long userId = request.getUserId();
        User user = userService.getUser(userId);
        // если пользователя нет, заводим представителя для client
        if (user == null) {
            Result<User> addUserResult = addUserService.addUserFromBlackbox(request.getUserId(), request.getClientId());
            if (!addUserResult.isSuccessful()) {
                logValidationErrors(request, addUserResult);
                return new IdmErrorResponse(mapDefectsToErrorCode(addUserResult));
            }
        }

        if (request.getRole() == ConnectIdmRoles.EMPLOYEE) {
            return new IdmSuccessResponse();
        }

        if (request.getRole() == ConnectIdmRoles.CHIEF) {
            // переключаем главного представителя.
            Result<User> changeChiefResult =
                    clientChiefService.changeChief(userId, request.getClientId());
            if (changeChiefResult.isSuccessful()) {
                return new IdmSuccessResponse();
            }
            logValidationErrors(request, changeChiefResult);
            return new IdmErrorResponse(mapDefectsToErrorCode(changeChiefResult));
        }
        // не должны сюда дойти
        throw new IllegalStateException("unexpected behaviour");
    }

    private void logValidationErrors(AddPersonalRoleRequest request, Result<User> addUserResult) {
        var defectInfos = addUserResult.getValidationResult().flattenErrors();
        String errors = defectsToString(translationService, defectPresentationRegistry, defectInfos);
        String requestInfo = String.format("AddPersonalRoleRequest{ClientId=%s, OrgId=%s, Role=%s, UserId=%s}",
                request.getClientId(), request.getOrgId(), request.getRole(), request.getUserId());
        String message = String.format("Got validation errors {%s} on request %s", errors, requestInfo);
        logger.warn(message);
    }

    private ClientId extractClientId(AddRoleRequest request) {
        Long resourceId = request.getFields().getResourceId();
        return ClientId.fromLong(resourceId);
    }

    // todo maxlog: тесты на валидацию
    private Optional<ConnectError> validate(AddRoleRequest request) {
        // проверим наличие полей
        if (request.getFields() == null || request.getFields().getResourceId() == null) {
            return Optional.of(ConnectError.NO_RESOURCE_ID);
        }
        if (request.getSubjectType() == null) {
            return Optional.of(ConnectError.NO_SUBJECT_TYPE);
        }
        if (isBlank(request.getPath())) {
            return Optional.of(ConnectError.NO_PATH);
        }
        if (request.getId() == null) {
            return Optional.of(ConnectError.NO_ID);
        }
        if (request.getOrgId() == null && request.getSubjectType() != SubjectType.ORGANIZATION) {
            // org_id опционален только для subject_type = ORGANIZATION
            return Optional.of(ConnectError.NO_ORG_ID);
        }

        // проверим валидность заданных полей
        if (!ConnectIdmRoles.isPathValid(request.getPath())) {
            return Optional.of(ConnectError.WRONG_PATH);
        }

        return Optional.empty();
    }
}
