package ru.yandex.direct.intapi.entity.connect.service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.http.NameValuePair;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.message.BasicNameValuePair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.intapi.entity.connect.controller.ConnectIdmRolesController;
import ru.yandex.direct.intapi.entity.connect.model.GetRolesResponse;
import ru.yandex.direct.intapi.entity.connect.model.GetRolesResponseItem;
import ru.yandex.direct.intapi.entity.connect.model.IdmErrorResponse;
import ru.yandex.direct.intapi.entity.connect.model.IdmResponse;

import static java.util.Collections.singleton;
import static java.util.function.Function.identity;
import static ru.yandex.direct.intapi.entity.connect.converter.ConnectIdmRolesResponseConverter.toAssociatedItem;
import static ru.yandex.direct.intapi.entity.connect.converter.ConnectIdmRolesResponseConverter.toChiefItem;
import static ru.yandex.direct.intapi.entity.connect.converter.ConnectIdmRolesResponseConverter.toEmployeeItem;
import static ru.yandex.direct.intapi.entity.connect.model.ConnectIdmConstants.RESOURCE_ID;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@Service
@ParametersAreNonnullByDefault
public class ConnectIdmGetPagedRolesService {
    public static final String LAST_SHARD = "last_shard";
    public static final String LAST_CLIENT_ID = "last_client_id";
    public static final String CLIENTS_LIMIT = "clients_limit";

    private static final int DEFAULT_CLIENTS_LIMIT = 100;
    private static final String EMPTY_NEXT_URL = null;

    private final ShardHelper shardHelper;
    private final ClientService clientService;
    private final UserService userService;

    @Autowired
    public ConnectIdmGetPagedRolesService(ShardHelper shardHelper, ClientService clientService,
                                          UserService userService) {
        this.shardHelper = shardHelper;
        this.clientService = clientService;
        this.userService = userService;
    }

    public IdmResponse getSingleClientRoles(ClientId clientId) {
        Client client1 = clientService.getClient(clientId);
        if (client1 == null) {
            return new IdmErrorResponse(String.format("%s='%s' not found", RESOURCE_ID, clientId));
        }
        List<Long> uids = userService.massGetUidsByClientIds(singleton(clientId)).get(clientId);
        List<GetRolesResponseItem> items = getRolesResponseItems(client1, uids);
        return new GetRolesResponse()
                .withNextUrl(EMPTY_NEXT_URL)
                .withRoles(items);
    }

    public IdmResponse getRolesPage(@Nullable ClientId lastClientId, @Nullable Integer lastShard,
                                    @Nullable Integer clientsLimit) {
        clientsLimit = nvl(clientsLimit, DEFAULT_CLIENTS_LIMIT);
        List<Client> clients = clientService.getNextPageConnectOrgClients(lastShard, lastClientId, clientsLimit);
        Map<Client, List<Long>> usersByClient = getUidsByClient(clients);
        List<GetRolesResponseItem> items = convertToRoleItems(usersByClient);
        String nextUrl = getNextUrl(clientsLimit, clients);
        return new GetRolesResponse()
                .withNextUrl(nextUrl)
                .withRoles(items);
    }

    private Map<Client, List<Long>> getUidsByClient(List<Client> clients) {
        Map<ClientId, Client> clientsByIds = StreamEx.of(clients)
                .mapToEntry(Client::getClientId, identity())
                .mapKeys(ClientId::fromLong)
                .toMap();
        Map<ClientId, List<Long>> uidsByClientIds = userService.massGetUidsByClientIds(clientsByIds.keySet());
        return EntryStream.of(uidsByClientIds)
                .mapKeys(clientsByIds::get)
                .toMap();
    }

    private List<GetRolesResponseItem> convertToRoleItems(Map<Client, List<Long>> uidsByClients) {
        return EntryStream.of(uidsByClients)
                .mapKeyValue(this::getRolesResponseItems)
                .flatMap(StreamEx::of)
                .toList();
    }

    /**
     * Возвращает список ролей для заданного клиента и uid-ов его представителей.
     * Первой возвращается роль chief для главного представителя, затем роли employee для всех представителей
     * отсортированные по Uid, затем роль associated для организации.
     * Uid главного представителя возвращается два раза, с ролями chief и employee.
     */
    private List<GetRolesResponseItem> getRolesResponseItems(Client client, List<Long> uids) {
        ArrayList<GetRolesResponseItem> items = new ArrayList<>();
        //chief
        items.add(toChiefItem(client));
        //employee
        List<GetRolesResponseItem> employeeItems = StreamEx.of(uids)
                .sorted()
                .map(uid -> toEmployeeItem(client, uid))
                .toList();
        items.addAll(employeeItems);
        //associated
        if (client.getConnectOrgId() != null) {
            items.add(toAssociatedItem(client));
        }
        return items;
    }

    private String getNextUrl(int clientsLimit, List<Client> clients) {
        if (clients.size() < clientsLimit) {
            return EMPTY_NEXT_URL;
        }
        Client last = Iterables.getLast(clients);
        int lastShard = shardHelper.getShardByClientId(ClientId.fromLong(last.getClientId()));
        List<NameValuePair> valuePairs = EntryStream
                .of(LAST_SHARD, lastShard,
                        LAST_CLIENT_ID, last.getClientId(),
                        CLIENTS_LIMIT, clientsLimit)
                .mapValues(Object::toString)
                .mapKeyValue(BasicNameValuePair::new)
                .map(NameValuePair.class::cast)
                .toList();
        URIBuilder ub = new URIBuilder()
                .setPath(ConnectIdmRolesController.GET_ROLES_BASE_PATH)
                .addParameters(valuePairs);
        return ub.toString();
    }

}
