package ru.yandex.direct.intapi.entity.connect.service;

import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.intapi.entity.connect.model.ConnectError;
import ru.yandex.direct.intapi.entity.connect.model.ConnectIdmRoles;
import ru.yandex.direct.intapi.entity.connect.model.IdmErrorResponse;
import ru.yandex.direct.intapi.entity.connect.model.IdmResponse;
import ru.yandex.direct.intapi.entity.connect.model.IdmSuccessResponse;
import ru.yandex.direct.intapi.entity.connect.model.RemovePersonalRoleRequest;
import ru.yandex.direct.intapi.entity.connect.model.RemoveRoleRequest;
import ru.yandex.direct.intapi.entity.connect.model.SubjectType;
import ru.yandex.direct.result.Result;

import static org.apache.commons.lang.StringUtils.isBlank;
import static ru.yandex.direct.intapi.entity.connect.model.DefectConvertor.mapDefectsToErrorCode;

@Service
@ParametersAreNonnullByDefault
public class ConnectIdmRemoveRoleService {

    private final ClientService clientService;
    private final UserService userService;

    public ConnectIdmRemoveRoleService(ClientService clientService, UserService userService) {
        this.clientService = clientService;
        this.userService = userService;
    }

    public IdmResponse removeRole(RemoveRoleRequest request) {
        Optional<ConnectError> error = validate(request);
        if (error.isPresent()) {
            return new IdmErrorResponse(error.get());
        }
        ClientId clientId = extractClientId(request);
        Client client = clientService.getClient(clientId);
        if (client == null) {
            return new IdmErrorResponse(ConnectError.RESOURCE_NOT_FOUND);
        }
        ConnectIdmRoles role = ConnectIdmRoles.getRoleByPath(request.getPath());
        if (role == ConnectIdmRoles.ASSOCIATED) {
            if (!request.getSubjectType().equals(SubjectType.ORGANIZATION)) {
                return new IdmErrorResponse(ConnectError.SUBJECT_TYPE_AND_ROLE_MISSMATCH);
            }
            return removeOrgRole(request.getId(), client.getConnectOrgId(), client.getClientId());
        }
        RemovePersonalRoleRequest personalRequest = new RemovePersonalRoleRequest()
                .withOrgId(request.getOrgId())
                .withClient(client)
                .withUserId(request.getId())
                .withRole(role);
        return removePersonalRole(personalRequest);
    }

    private IdmResponse removeOrgRole(Long reqOrgId, Long clientOrgId, Long clientId) {
        if (!reqOrgId.equals(clientOrgId)) {
            return new IdmErrorResponse(ConnectError.RESOURCE_NOT_ASSOCIATED);
        }
        clientService.updateClientConnectOrgId(clientId, null);
        return new IdmSuccessResponse();
    }

    private IdmResponse removePersonalRole(RemovePersonalRoleRequest request) {
        Client client = request.getClient();
        if (!request.getOrgId().equals(client.getConnectOrgId())) {
            return new IdmErrorResponse(ConnectError.RESOURCE_NOT_ASSOCIATED);
        }

        User user = userService.getUser(request.getUserId());
        if (user != null) {
            if (!client.getClientId().equals(user.getClientId().asLong())) {
                return new IdmErrorResponse(ConnectError.USER_ASSOCIATED_TO_ANOTHER_RESOURCE);
            }
            if (request.getRole() == ConnectIdmRoles.CHIEF) {
                return new IdmSuccessResponse();
            }
        }

        //Сюда доходим если либо запрошен корректный отзыв роли employee,
        //либо передан uid несуществующего пользователя. В последнем случае dropClientRep вернет ошибку
        Result<User> result = userService
                .dropClientRep(user);

        if (result.isSuccessful()) {
            return new IdmSuccessResponse();
        } else {
            return new IdmErrorResponse(mapDefectsToErrorCode(result));
        }
    }

    private ClientId extractClientId(RemoveRoleRequest request) {
        Long resourceId = request.getFields().getResourceId();
        return ClientId.fromLong(resourceId);
    }

    private Optional<ConnectError> validate(RemoveRoleRequest request) {
        if (request.getFields() == null || request.getFields().getResourceId() == null) {
            return Optional.of(ConnectError.NO_RESOURCE_ID);
        }
        if (request.getSubjectType() == null) {
            return Optional.of(ConnectError.NO_SUBJECT_TYPE);
        }
        if (isBlank(request.getPath())) {
            return Optional.of(ConnectError.NO_PATH);
        }
        if (request.getId() == null) {
            return Optional.of(ConnectError.NO_ID);
        }
        if (request.getOrgId() == null && request.getSubjectType() != SubjectType.ORGANIZATION) {
            return Optional.of(ConnectError.NO_ORG_ID);
        }
        if (!ConnectIdmRoles.isPathValid(request.getPath())) {
            return Optional.of(ConnectError.WRONG_PATH);
        }
        return Optional.empty();
    }
}
