package ru.yandex.direct.intapi.entity.copyentity.controller;

import java.util.Collections;
import java.util.List;
import java.util.Objects;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.core.copyentity.CopyConfig;
import ru.yandex.direct.core.copyentity.CopyConfigBuilder;
import ru.yandex.direct.core.copyentity.CopyLocker;
import ru.yandex.direct.core.copyentity.CopyOperation;
import ru.yandex.direct.core.copyentity.CopyOperationFactory;
import ru.yandex.direct.core.copyentity.CopyResult;
import ru.yandex.direct.core.copyentity.model.CampaignCopyJobParams;
import ru.yandex.direct.core.copyentity.model.CopyCampaignFlags;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.service.CopyCampaignService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.IntApiException;
import ru.yandex.direct.intapi.entity.copyentity.model.CopyCampaignRequest;
import ru.yandex.direct.intapi.entity.copyentity.model.CopyClientCampaignBulkResponse;
import ru.yandex.direct.intapi.entity.copyentity.model.CopyClientCampaignResponse;
import ru.yandex.direct.intapi.logging.ClientIdParam;
import ru.yandex.direct.intapi.logging.OperatorUid;
import ru.yandex.direct.intapi.validation.kernel.ValidationResultConversionService;
import ru.yandex.direct.intapi.validation.model.IntapiError;
import ru.yandex.direct.intapi.validation.model.IntapiSuccessResponse;
import ru.yandex.direct.intapi.validation.model.IntapiValidationResult;
import ru.yandex.direct.redislock.DistributedLock;
import ru.yandex.direct.tvm.AllowServices;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.model.WebResponse;

import static java.util.Objects.requireNonNull;
import static ru.yandex.direct.feature.FeatureName.INTERCLIENT_CAMPAIGN_COPY_ALLOWED;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.DIRECT_INTAPI_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_INTAPI_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_TEST;

@RestController
@RequestMapping(value = "copy_entity",
        consumes = MediaType.APPLICATION_JSON_UTF8_VALUE,
        produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Api(tags = "copyEntity")
@AllowServices(production = {DIRECT_WEB_PROD, DIRECT_INTAPI_PROD},
        testing = {DIRECT_DEVELOPER, DIRECT_WEB_TEST, DIRECT_INTAPI_TEST})
public class CopyEntityController {
    private static final Logger logger = LoggerFactory.getLogger(CopyEntityController.class);

    private final ShardHelper shardHelper;
    private final FeatureService featureService;
    private final CopyOperationFactory copyOperationFactory;
    private final ValidationResultConversionService validationResultConversionService;
    private final CopyLocker locker;
    private final CopyCampaignService copyCampaignService;

    @Autowired
    @SuppressWarnings("checkstyle:parameternumber")
    public CopyEntityController(ShardHelper shardHelper,
                                FeatureService featureService,
                                CopyOperationFactory copyOperationFactory,
                                ValidationResultConversionService validationResultConversionService,
                                CopyLocker locker,
                                CopyCampaignService copyCampaignService) {
        this.shardHelper = shardHelper;
        this.featureService = featureService;
        this.copyOperationFactory = copyOperationFactory;
        this.validationResultConversionService = validationResultConversionService;
        this.locker = locker;
        this.copyCampaignService = copyCampaignService;
    }

    @ApiOperation(
            value = "copyClientCampaigns",
            httpMethod = "GET",
            nickname = "copyClientCampaigns"
    )
    @ApiResponses({
            @ApiResponse(code = 400, message = "Bad params", response = ErrorResponse.class),
            @ApiResponse(code = 200, message = "Ok", response = IntapiSuccessResponse.class)
    })
    @GetMapping(path = "/client_campaign",
            consumes = MediaType.APPLICATION_JSON_UTF8_VALUE,
            produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @ResponseBody
    @SuppressWarnings("unchecked")
    public WebResponse copyClientCampaign(@RequestParam(value = "operator_uid") @OperatorUid Long operatorUid,
                                          @RequestParam(value = "client_id") @ClientIdParam Long clientId,
                                          @RequestParam(value = "campaign_id") Long campaignId) {
        if (campaignId == null) {
            throwBadParamError("campaign_id is null");
        }
        if (operatorUid == null) {
            throwBadParamError("operator_uid is null");
        }
        if (clientId == null) {
            throwBadParamError("client_id is null");
        }


        DistributedLock lock = locker.create(requireNonNull(clientId).toString());
        try {
            if (!lock.lock()) {
                return new CopyClientCampaignResponse(
                        new IntapiValidationResult(List.of(new IntapiError().withCode("FAILED_TO_ACQUIRE_LOCK"))),
                        null);
            }

            ClientId clientIdObj = ClientId.fromLong(clientId);
            CopyConfig copyConfig = new CopyConfigBuilder(
                    clientIdObj, clientIdObj, Objects.requireNonNull(operatorUid),
                    BaseCampaign.class, List.of(requireNonNull(campaignId))
            ).build();

            CopyOperation operation = copyOperationFactory.build(copyConfig);

            CopyResult<Long> copyResult = operation.copy();
            copyResult.logFailedResultsForMonitoring();

            ValidationResult<?, Defect> validationResult = copyResult.getMassResult().getValidationResult();
            Long newCampaignId = (Long) copyResult.getEntityMapping(BaseCampaign.class).get(campaignId);
            return new CopyClientCampaignResponse(
                    validationResultConversionService.buildValidationResponse(validationResult),
                    newCampaignId);
        } catch (InterruptedException ex) {
            logger.warn("Interrupted while acquiring lock.");
            Thread.currentThread().interrupt();
            return new CopyClientCampaignResponse(
                    new IntapiValidationResult(List.of(new IntapiError().withCode("FAILED_TO_ACQUIRE_LOCK"))),
                    0L);
        } finally {
            if (lock.isLocked()) {
                lock.unlock();
            }
        }
    }

    @ApiOperation(
            value = "copyClientCampaigns",
            httpMethod = "POST",
            nickname = "copyClientCampaigns"
    )
    @ApiResponses({
            @ApiResponse(code = 400, message = "Bad params", response = ErrorResponse.class),
            @ApiResponse(code = 200, message = "Ok", response = IntapiSuccessResponse.class)
    })
    @PostMapping(path = "/client_campaigns",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseBody
    public CopyClientCampaignBulkResponse copyClientCampaigns(@RequestBody CopyCampaignRequest request) {
        if (request.getCampaignIds() == null) {
            throwBadParamError("campaign_ids is null");
        } else if (request.getOperatorUid() == null) {
            throwBadParamError("operator_uid is null");
        } else if (request.getClientIdFrom() == null) {
            throwBadParamError("client_id_from is null");
        } else if (request.getClientIdTo() == null) {
            throwBadParamError("client_id_to is null");
        } else if (request.getFlags() == null) {
            throwBadParamError("flags is null");
        }

        List<Long> campaignIds = request.getCampaignIds();
        var clientIdFrom = ClientId.fromLong(request.getClientIdFrom());
        var clientIdTo = ClientId.fromLong(request.getClientIdTo());
        var operatorUid = requireNonNull(request.getOperatorUid(), "operator_uid is null");

        if (!clientIdTo.equals(clientIdFrom) && isInterclientCopingDeniedForOperator(operatorUid)) {
            throwInterclientCopingDeniedForOperator();
        }

        var shard = shardHelper.getShardByClientIdStrictly(clientIdFrom);
        var flags = request.getFlags();
        List<CampaignCopyJobParams> params = StreamEx.of(campaignIds)
                .map(campaignId -> new CampaignCopyJobParams(clientIdFrom, clientIdTo, campaignId, flags, null, null))
                .toList();
        copyCampaignService.createJobs(shard, request.getOperatorUid(), clientIdFrom, params);

        return new CopyClientCampaignBulkResponse(campaignIds, 1L);
    }

    /**
     * Ручка для тестирования.
     * https://st.yandex-team.ru/DIRECT-113932#5e56305a68b96f71d775e52a
     */
    @ApiOperation(
            value = "copyCampaigns",
            httpMethod = "POST",
            nickname = "copyCampaigns"
    )
    @ApiResponses({
            @ApiResponse(code = 400, message = "Bad params", response = ErrorResponse.class),
            @ApiResponse(code = 200, message = "Ok", response = IntapiSuccessResponse.class)
    })
    @PostMapping(path = "/campaign",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseBody
    @SuppressWarnings("unchecked")
    public WebResponse copyCampaign(@RequestBody CopyCampaignRequest request) {
        List<Long> campaignIds = requireNonNull(request.getCampaignIds());
        if (campaignIds.size() != 1) {
            throwBadParamError("campaign_id must contain one id.");
        }
        var campaignId = campaignIds.get(0);
        var operatorUid = requireNonNull(request.getOperatorUid(), "operator_uid is null");
        var clientIdFrom = requireNonNull(request.getClientIdFrom(), "client_id_from is null");
        var clientIdTo = requireNonNull(request.getClientIdTo(), "client_id_to is null");

        if (!clientIdTo.equals(clientIdFrom) && isInterclientCopingDeniedForOperator(operatorUid)) {
            throwInterclientCopingDeniedForOperator();
        }

        var copyResult = copyCampaignService.copyCampaigns(
                ClientId.fromLong(clientIdFrom),
                ClientId.fromLong(clientIdTo),
                operatorUid,
                List.of(campaignId), new CopyCampaignFlags.Builder()
                        .withCopyKeywordStatuses(true)
                        .withCopyArchived(true)
                        .withCopyStopped(true)
                        .withCopyBannerStatuses(true)
                        .build());

        copyResult.logFailedResultsForMonitoring();

        var validationResult = copyResult.getMassResult().getValidationResult();
        Long newCampaignId = (Long) copyResult.getEntityMapping(BaseCampaign.class).get(campaignId);
        return new CopyClientCampaignResponse(
                validationResultConversionService.buildValidationResponse(validationResult),
                newCampaignId);
    }

    /**
     * Ручка для тестирования копирования между клиентами.
     * Как /campaign копирует кампанию в синхронном режиме, без постановки в очередь
     */
    @ApiOperation(
            value = "copyCampaignsInterClient",
            httpMethod = "POST",
            nickname = "copyCampaignsInterClient"
    )
    @ApiResponses({
            @ApiResponse(code = 400, message = "Bad params", response = ErrorResponse.class),
            @ApiResponse(code = 200, message = "Ok", response = IntapiSuccessResponse.class)
    })
    @PostMapping(path = "/interclient/campaign",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseBody
    @SuppressWarnings("unchecked")
    public WebResponse copyCampaignInterClient(@RequestBody CopyCampaignRequest request) {
        List<Long> campaignIds = requireNonNull(request.getCampaignIds());
        if (campaignIds.size() != 1) {
            throwBadParamError("campaign_id must contain one id.");
        }
        Long campaignId = campaignIds.get(0);
        Long operatorUid = requireNonNull(request.getOperatorUid(), "operator_uid is null");
        Long clientIdFrom = requireNonNull(request.getClientIdFrom(), "client_id_from is null");
        Long clientIdTo = requireNonNull(request.getClientIdTo(), "client_id_to is null");

        if (!clientIdTo.equals(clientIdFrom) && isInterclientCopingDeniedForOperator(operatorUid)) {
            throwInterclientCopingDeniedForOperator();
        }

        CopyResult<Long> copyResult = copyCampaignService.copyCampaigns(
                ClientId.fromLong(clientIdFrom),
                ClientId.fromLong(clientIdTo),
                operatorUid,
                List.of(campaignId), new CopyCampaignFlags.Builder()
                        .withCopyKeywordStatuses(true)
                        .withCopyArchived(true)
                        .withCopyStopped(true)
                        .withCopyBannerStatuses(true)
                        .build());

        copyResult.logFailedResultsForMonitoring();

        var validationResult = copyResult.getMassResult().getValidationResult();
        Long newCampaignId = (Long) copyResult.getEntityMapping(BaseCampaign.class).get(campaignId);
        return new CopyClientCampaignResponse(
                validationResultConversionService.buildValidationResponse(validationResult),
                newCampaignId);
    }

    private boolean isInterclientCopingDeniedForOperator(Long operatorUid) {
        return !featureService.isEnabledForUid(operatorUid,
                Collections.singletonList(INTERCLIENT_CAMPAIGN_COPY_ALLOWED));
    }

    private void throwBadParamError(String message) {
        var errorResponse = new ErrorResponse(ErrorResponse.ErrorCode.BAD_PARAM, message);
        throw new IntApiException(HttpStatus.BAD_REQUEST, errorResponse);
    }

    private void throwInterclientCopingDeniedForOperator() {
        var errorResponse = new ErrorResponse(ErrorResponse.ErrorCode.PERMISSION_DENIED,
                "Feature " + INTERCLIENT_CAMPAIGN_COPY_ALLOWED.getName() + " disabled for operator");
        throw new IntApiException(HttpStatus.FORBIDDEN, errorResponse);
    }
}
