package ru.yandex.direct.intapi.entity.creative;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.ws.rs.POST;
import javax.ws.rs.core.MediaType;

import io.swagger.annotations.Api;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.service.BannerService;
import ru.yandex.direct.core.entity.creative.model.UpdateSmartAdGroupCreativeGroups;
import ru.yandex.direct.core.entity.creative.model.UpdateSmartAdGroupCreativeGroupsItem;
import ru.yandex.direct.core.entity.creative.service.BannerStorageCreativeService;
import ru.yandex.direct.core.entity.creative.service.BannerStorageCreativeValidationService;
import ru.yandex.direct.core.entity.creative.service.CreativeService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.intapi.entity.creative.model.UpdateCreativeGroups;
import ru.yandex.direct.intapi.entity.creative.model.UpdateSmartAdGroupCreativeGroupsPayloadItem;
import ru.yandex.direct.intapi.entity.creative.model.UpdateSmartCreativeGroupResponse;
import ru.yandex.direct.intapi.validation.kernel.ValidationResultConversionService;
import ru.yandex.direct.intapi.validation.model.IntapiResponse;
import ru.yandex.direct.tvm.AllowServices;

import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.DIRECT_INTAPI_TEST;
import static ru.yandex.direct.tvm.TvmService.GEO_PRODUCT_PROD;
import static ru.yandex.direct.tvm.TvmService.GEO_PRODUCT_TEST;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@RestController
@RequestMapping("creative")
@Api(tags = "creative", value = "временно для создания смартовых групп креативов не из интерфейса")
@AllowServices(
        production = {GEO_PRODUCT_PROD},
        testing = {GEO_PRODUCT_TEST, DIRECT_DEVELOPER, DIRECT_INTAPI_TEST})
public class CreativeController {

    private final CreativeService creativeService;
    private final BannerStorageCreativeValidationService bannerStorageCreativeValidationService;
    private final ValidationResultConversionService validationResultConversionService;
    private final BannerStorageCreativeService bannerStorageCreativeService;
    private final BannerService bannerService;

    @Autowired
    public CreativeController(
            CreativeService creativeService,
            BannerStorageCreativeValidationService bannerStorageCreativeValidationService,
            ValidationResultConversionService validationResultConversionService,
            BannerStorageCreativeService bannerStorageCreativeService,
            BannerService bannerService) {
        this.creativeService = creativeService;
        this.bannerStorageCreativeValidationService = bannerStorageCreativeValidationService;
        this.validationResultConversionService = validationResultConversionService;
        this.bannerStorageCreativeService = bannerStorageCreativeService;
        this.bannerService = bannerService;
    }

    /**
     * Обновление групп креативов, привязанных к перформанс-группам объявлений
     * (может затронуть другие группы объявлений - если они используют те же группы креативов, что и обновляемые).
     * В случае отсутствия привязанных групп креативов для группы, создает группу креативов с заданными параметрами,
     * а затем создает по баннеру для каждого креатива в группе.
     * Валидируется, что нет групп креативов, связанных со сразу несколькими обновляемыми группами объявлений.
     */
    @POST
    @RequestMapping(path = "update_smart_creative_groups",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON,
            produces = MediaType.APPLICATION_JSON
    )
    public IntapiResponse updateSmartAdGroupCreativeGroups(
            @RequestParam("client_id") long clientId,
            @RequestParam("operator_uid") long operatorUid,
            @RequestBody UpdateCreativeGroups request) {

        ClientId clientId1 = ClientId.fromLong(clientId);
        if (isEmpty(request.getUpdateItems())) {
            return new UpdateSmartCreativeGroupResponse(true, Collections.emptyList());
        }

        UpdateSmartAdGroupCreativeGroups input = new UpdateSmartAdGroupCreativeGroups()
                .withSaveDraft(request.getSaveDraft())
                .withUpdateItems(mapList(request.getUpdateItems(), item -> new UpdateSmartAdGroupCreativeGroupsItem()
                        .withAdGroupId(item.getAdGroupId())
                        .withTheme(item.getTheme())
                        .withDomain(item.getDomain())
                        .withLogoFileId(item.getLogoFileId())));

        var adGroupIds = mapList(input.getUpdateItems(), UpdateSmartAdGroupCreativeGroupsItem::getAdGroupId);
        var creativeByGroupId = creativeService.getCreativesByPerformanceAdGroups(clientId1, adGroupIds);

        var validationResult = bannerStorageCreativeValidationService
                .validateAdGroupCreativeGroupsUpdateRequest(input, clientId1, creativeByGroupId);
        if (validationResult.hasAnyErrors()) {
            return validationResultConversionService.buildValidationResponse(validationResult);
        }

        var creativeGroupIdsByAdGroupId = EntryStream.of(creativeByGroupId)
                .mapValues(creatives -> mapAndFilterToSet(creatives,
                        ru.yandex.direct.core.entity.creative.model.Creative::getCreativeGroupId,
                        Objects::nonNull))
                .toMap();

        boolean saveDraft = nvl(input.getSaveDraft(), false);
        List<Long> successfullyUpdatedAdGroupIds = StreamEx.of(input.getUpdateItems())
                .filter(item -> bannerStorageCreativeService.updateSmartAdGroupCreativeGroupsItem(clientId1,
                        operatorUid, saveDraft, item, creativeGroupIdsByAdGroupId.get(item.getAdGroupId())))
                .map(UpdateSmartAdGroupCreativeGroupsItem::getAdGroupId)
                .toList();

        Map<Long, List<Long>> bannersByAdGroupIds =
                EntryStream.of(bannerService.getBannersByAdGroupIds(successfullyUpdatedAdGroupIds))
                        .mapValues(banners -> mapList(banners, BannerWithSystemFields::getId))
                        .toMap();

        var successfullyUpdatedAdGroups = StreamEx.of(successfullyUpdatedAdGroupIds)
                .map(adGroupId -> new UpdateSmartAdGroupCreativeGroupsPayloadItem()
                        .withAdGroupId(adGroupId)
                        .withBannerIds(bannersByAdGroupIds.get(adGroupId)))
                .collect(Collectors.toList());

        return new UpdateSmartCreativeGroupResponse(true, successfullyUpdatedAdGroups);
    }

}
