package ru.yandex.direct.intapi.entity.crm;

import java.util.List;

import javax.annotation.Nonnull;
import javax.ws.rs.core.MediaType;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.entity.crm.model.CrmAdgroupInfo;
import ru.yandex.direct.intapi.entity.crm.model.FallbackEnableDetails;
import ru.yandex.direct.intapi.entity.crm.model.FallbackStatusInfo;
import ru.yandex.direct.intapi.validation.model.IntapiResponse;
import ru.yandex.direct.intapi.validation.model.IntapiSuccessResponse;
import ru.yandex.direct.tvm.AllowServices;
import ru.yandex.direct.web.core.model.WebResponse;

import static ru.yandex.direct.tvm.TvmService.CRM_PROD;
import static ru.yandex.direct.tvm.TvmService.CRM_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;

/**
 * Сборник intapi-ручек для CRM
 */
@RestController
@RequestMapping("crm")
@Api(value = "Ручки для CRM")
@AllowServices(production = {CRM_PROD}, testing = {CRM_TEST, DIRECT_DEVELOPER})
public class CrmController {
    private final CrmService crmService;

    @Autowired
    public CrmController(CrmService crmService) {
        this.crmService = crmService;
    }

    /**
     * По списку adgroupid получить id клиентов и кампаний.
     * Используется из интерфейса CRM - найти группу
     */
    @ApiOperation(
            value = "Получение данных о adgroup-ах по id.",
            httpMethod = "POST",
            nickname = "getAdgroupsInfo"
    )
    @RequestMapping(path = "adgroups",
            method = RequestMethod.POST,
            produces = MediaType.APPLICATION_JSON
    )
    public List<CrmAdgroupInfo> getAdgroups(
            @ApiParam(value = "id adgroup-ы", required = true)
            @RequestBody @Nonnull List<Long> adgroupIds
    ) {
        return crmService.getAdgroups(adgroupIds);
    }

    /**
     * Включение fallback-режима для пинкодницы.
     * Управляется через property enable_limited_support_read_all_clients.
     * В fallback-режиме операторы с ролью limited_support имеют ro доступ ко всем клиентам,
     * а не только к связанным с ними.
     */
    @ApiOperation(
            value = "Включение fallback-режима для пинкодницы",
            httpMethod = "POST",
            nickname = "enable"
    )
    @ApiResponses({
            @ApiResponse(code = 400, message = "Bad params", response = ErrorResponse.class),
            @ApiResponse(code = 200, message = "Ok", response = IntapiSuccessResponse.class)
    })

    @RequestMapping(path = "fallback/enable",
            method = RequestMethod.POST,
            produces = MediaType.APPLICATION_JSON
    )
    public WebResponse fallbackEnable(
            @ApiParam(value = "Метаданные включения", required = true)
            @RequestBody @Nonnull FallbackEnableDetails data
    ) {
        return crmService.fallbackEnable(data);
    }

    /**
     * Выключение fallback-режима для пинкодницы.
     */
    @ApiOperation(
            value = "Выключение fallback-режима для пинкодницы",
            httpMethod = "POST",
            nickname = "disable"
    )
    @RequestMapping(path = "fallback/disable",
            method = RequestMethod.POST,
            produces = MediaType.APPLICATION_JSON
    )
    public IntapiResponse fallbackDisable() {
        crmService.fallbackDisable();
        return new IntapiSuccessResponse();
    }

    /**
     * Получение текущего статуса fallback-режима для пинкодницы.
     */
    @ApiOperation(
            value = "Текущий статус fallback-режима",
            httpMethod = "GET",
            nickname = "status"
    )
    @RequestMapping(path = "fallback/status",
            method = RequestMethod.GET,
            produces = MediaType.APPLICATION_JSON
    )
    public FallbackStatusInfo fallbackStatus() {
        return crmService.getFallbackStatus();
    }
}
