package ru.yandex.direct.intapi.entity.crm;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.YesNo;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum;
import ru.yandex.direct.dbutil.QueryWithForbiddenShardMapping;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.intapi.entity.crm.model.CrmAdgroupInfo;
import ru.yandex.direct.intapi.entity.crm.model.FallbackEnableDetails;
import ru.yandex.direct.intapi.entity.crm.model.FallbackStatusInfo;
import ru.yandex.direct.intapi.validation.model.IntapiSuccessResponse;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.model.WebResponse;

import static java.util.Objects.isNull;
import static java.util.Objects.nonNull;

@Service
@ParametersAreNonnullByDefault
public class CrmService {
    private final AdGroupRepository adGroupRepository;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final ShardHelper shardHelper;

    @Autowired
    public CrmService(
            AdGroupRepository adGroupRepository,
            PpcPropertiesSupport ppcPropertiesSupport,
            ShardHelper shardHelper
    ) {
        this.adGroupRepository = adGroupRepository;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.shardHelper = shardHelper;
    }

    @QueryWithForbiddenShardMapping("В ручку crm/adgroups приходит только adgroup id")
    public List<CrmAdgroupInfo> getAdgroups(List<Long> adgroupIds) {
        Map<Long, AdGroup> data = shardHelper.groupByShard(adgroupIds, ShardKey.PID)
                .stream()
                .mapKeyValue((shard, ids) -> adGroupRepository.getAdGroups(shard, ids))
                .flatMap(Collection::stream)
                .toMap(AdGroup::getId, g -> g);

        List<Long> cids = data.values().stream().map(AdGroup::getCampaignId).distinct().collect(Collectors.toList());
        Map<Long, Long> clientIdsByCampaignIds = shardHelper.getClientIdsByCampaignIds(cids);

        ArrayList<CrmAdgroupInfo> ret = new ArrayList<>();
        for (Long adgroupId : adgroupIds) {
            AdGroup adgroup = data.get(adgroupId);
            Optional<ClientId> clientId = Optional.ofNullable(adgroup)
                    .map(AdGroup::getCampaignId)
                    .map(clientIdsByCampaignIds::get)
                    .map(ClientId::fromLong);
            if (clientId.isPresent()) {
                ret.add(CrmAdgroupInfo.create(adgroupId, clientId.get(), adgroup.getCampaignId()));
            } else {
                ret.add(CrmAdgroupInfo.error(adgroupId, "Not found"));
            }
        }

        return ret;
    }

    public WebResponse fallbackEnable(FallbackEnableDetails data) {
        if (isFallbackEnabled()) {
            return new WebErrorResponse("OPERATION_FAILED", "fallback already enabled");
        }
        ppcPropertiesSupport
                .set(PpcPropertyEnum.ENABLE_LIMITED_SUPPORT_READ_ALL_CLIENTS.getName(), "1");
        ppcPropertiesSupport
                .set(PpcPropertyEnum.ENABLE_LIMITED_SUPPORT_READ_ALL_CLIENTS_META.getName(), JsonUtils.toJson(data));

        return new IntapiSuccessResponse();
    }

    public void fallbackDisable() {
        ppcPropertiesSupport.set(PpcPropertyEnum.ENABLE_LIMITED_SUPPORT_READ_ALL_CLIENTS.getName(), "0");
        ppcPropertiesSupport.remove(PpcPropertyEnum.ENABLE_LIMITED_SUPPORT_READ_ALL_CLIENTS_META.getName());
    }

    public FallbackStatusInfo getFallbackStatus() {

        var result = new FallbackStatusInfo();
        if (isFallbackEnabled()) {
            result.setFallbackEnabled(YesNo.YES);
            result.setDetails(getFallbackDetails());
        } else {
            result.setFallbackEnabled(YesNo.NO);
        }

        return result;
    }

    private boolean isFallbackEnabled() {
        String fallbackEnabled =
                ppcPropertiesSupport.get(PpcPropertyEnum.ENABLE_LIMITED_SUPPORT_READ_ALL_CLIENTS.getName());
        return nonNull(fallbackEnabled) && Integer.parseInt(fallbackEnabled) > 0;
    }

    private FallbackEnableDetails getFallbackDetails() {
        String fallbackMeta =
                ppcPropertiesSupport.get(PpcPropertyEnum.ENABLE_LIMITED_SUPPORT_READ_ALL_CLIENTS_META.getName());
        if (isNull(fallbackMeta) || fallbackMeta.isEmpty()) {
            return null;
        }
        return JsonUtils.fromJson(fallbackMeta, FallbackEnableDetails.class);
    }
}
