package ru.yandex.direct.intapi.entity.daas.handle;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.protobuf.InvalidProtocolBufferException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import springfox.documentation.annotations.ApiIgnore;

import ru.yandex.direct.common.db.PpcPropertyName;
import ru.yandex.direct.core.entity.daas.DaasTaskType;
import ru.yandex.direct.intapi.entity.daas.ClientStatus;
import ru.yandex.direct.intapi.entity.daas.DaasAvailabilityRequest;
import ru.yandex.direct.intapi.entity.daas.DaasAvailabilityResult;
import ru.yandex.direct.intapi.entity.daas.ItemStatus;
import ru.yandex.direct.intapi.entity.daas.ResponseItem;
import ru.yandex.direct.tvm.AllowServices;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.common.db.PpcPropertyType.LONG_SET;
import static ru.yandex.direct.tvm.TvmService.COLLECTIONS_PROD;
import static ru.yandex.direct.tvm.TvmService.COLLECTIONS_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.TYCOON_FRONTEND_PROD;
import static ru.yandex.direct.tvm.TvmService.TYCOON_FRONTEND_TEST;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@ApiIgnore
@RestController
@RequestMapping(
        name = "daas",
        path = "/daas")
public class DaasIntapiController {
    public static final String PROTOBUF_MEDIA_TYPE = "application/x-protobuf";
    private static final Set<InputType> INPUT_TYPES_UNDER_PROPERTY = Set.of(InputType.collections,
            InputType.organizations);
    private static final Logger logger = LoggerFactory.getLogger(DaasIntapiController.class);

    @Autowired
    DaasIntapiController() {
    }

    @PostMapping(path = "get_client_item_statuses/{type}", consumes = PROTOBUF_MEDIA_TYPE,
            produces = PROTOBUF_MEDIA_TYPE)
    @AllowServices(production = {COLLECTIONS_PROD, TYCOON_FRONTEND_PROD},
            testing = {COLLECTIONS_TEST, TYCOON_FRONTEND_TEST, DIRECT_DEVELOPER})
    public DaasAvailabilityResult getStatuses(
            @PathVariable("type") InputType type,
            @RequestBody DaasAvailabilityRequest request) throws InvalidProtocolBufferException {
        DaasAvailabilityResult result = buildResult(request.getExternalIdsList());
        String responseJson = JsonUtils.PROTO_PRINTER.includingDefaultValueFields().print(result);
        logger.info("Get client item statuses response: {}", responseJson);
        return result;
    }

    @PostMapping(path = "get_client_item_statuses/json/{type}", consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE)
    @AllowServices(production = {COLLECTIONS_PROD, TYCOON_FRONTEND_PROD},
            testing = {COLLECTIONS_TEST, TYCOON_FRONTEND_TEST, DIRECT_DEVELOPER})
    public String getStatusesJson(
            @PathVariable("type") InputType type,
            @RequestBody DaasAvailabilityRequest request) throws InvalidProtocolBufferException {
        DaasAvailabilityResult result = buildResult(request.getExternalIdsList());
        String responseJson = JsonUtils.PROTO_PRINTER.includingDefaultValueFields().print(result);
        logger.info("Get client item statuses response: {}", responseJson);
        return responseJson;
    }

    private DaasAvailabilityResult buildResult(List<String> externalIds) {
        List<ResponseItem> responseItems = nvl(externalIds, new ArrayList<String>())
                .stream()
                .map(t -> ResponseItem.newBuilder()
                        .setItemStatus(ItemStatus.FORBIDDEN)
                        .setExternalId(t)
                        .build())
                .collect(Collectors.toList());
        return DaasAvailabilityResult.newBuilder()
                .setClientStatus(ClientStatus.FORBIDDEN_CLIENT)
                .addAllResponseItems(responseItems)
                .build();
    }

    enum InputType { //добавить тест, что мапится однозначно на модели типа в ядре
        collections(DaasTaskType.COLLECTION),
        organizations(DaasTaskType.ORGANIZATION);

        private DaasTaskType daasTaskType;

        InputType(DaasTaskType daasTaskType) {
            this.daasTaskType = daasTaskType;
        }

        public DaasTaskType getValue() {
            return daasTaskType;
        }
    }

    static PpcPropertyName<Set<Long>> getPropertyNameFromInputType(InputType inputType) {
        String ppcPropertyNameEnumString = "daas_" + inputType.name() + "_uids";
        return new PpcPropertyName(ppcPropertyNameEnumString, LONG_SET);
    }
}

