package ru.yandex.direct.intapi.entity.display.canvas;

import java.util.List;
import java.util.Map;

import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.core.MediaType;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.common.logging.EventType;
import ru.yandex.direct.common.logging.LoggingConfig;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.entity.display.canvas.model.CreativeCampaignRequest;
import ru.yandex.direct.intapi.entity.display.canvas.model.CreativeCampaignResult;
import ru.yandex.direct.intapi.entity.display.canvas.model.CreativeUploadData;
import ru.yandex.direct.intapi.entity.display.canvas.model.CreativeUploadResponse;
import ru.yandex.direct.intapi.entity.display.canvas.model.GetUsedCreativesRequest;
import ru.yandex.direct.intapi.entity.display.canvas.model.GetUsedCreativesResponse;
import ru.yandex.direct.intapi.entity.display.canvas.model.GetUsedCreativesSort;
import ru.yandex.direct.intapi.entity.display.canvas.model.GetUsedCreativesType;
import ru.yandex.direct.intapi.entity.display.canvas.model.Privileges;
import ru.yandex.direct.intapi.entity.display.canvas.service.DisplayCanvasAuthService;
import ru.yandex.direct.intapi.entity.display.canvas.service.DisplayCanvasCampaignService;
import ru.yandex.direct.intapi.entity.display.canvas.service.DisplayCanvasUploadService;
import ru.yandex.direct.intapi.entity.display.canvas.service.DisplayCanvasUsedCreativesService;
import ru.yandex.direct.intapi.logging.ClientIdParam;
import ru.yandex.direct.intapi.logging.OperatorUid;
import ru.yandex.direct.tvm.AllowServices;

import static ru.yandex.direct.tvm.TvmService.DIRECT_AUTOTESTS;
import static ru.yandex.direct.tvm.TvmService.DIRECT_CANVAS_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_CANVAS_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.DIRECT_SCRIPTS_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_SCRIPTS_TEST;
import static ru.yandex.direct.tvm.TvmService.LPC_SWITCHCONSTRUCTOR_PROD;
import static ru.yandex.direct.tvm.TvmService.LPC_SWITCHCONSTRUCTOR_TEST;
import static ru.yandex.direct.tvm.TvmService.SURVEYS_PROD;
import static ru.yandex.direct.tvm.TvmService.SURVEYS_TEST;

@RestController
@RequestMapping("DisplayCanvas")
@Api(value = "API для конструктора креативов")
@AllowServices(testing = {DIRECT_AUTOTESTS, DIRECT_DEVELOPER,
                            DIRECT_CANVAS_TEST, DIRECT_SCRIPTS_TEST, SURVEYS_TEST, LPC_SWITCHCONSTRUCTOR_TEST},
        production = {DIRECT_CANVAS_PROD, DIRECT_SCRIPTS_PROD, SURVEYS_PROD, LPC_SWITCHCONSTRUCTOR_PROD})
public class DisplayCanvasController {

    private final DisplayCanvasUploadService displayCanvasUploadService;
    private final DisplayCanvasAuthService displayCanvasAuthService;
    private final DisplayCanvasCampaignService displayCanvasCampaignService;
    private final DisplayCanvasUsedCreativesService displayCanvasUsedCreativesService;

    @Autowired
    public DisplayCanvasController(
            DisplayCanvasUploadService displayCanvasUploadService,
            DisplayCanvasAuthService displayCanvasAuthService,
            DisplayCanvasCampaignService displayCanvasCampaignService,
            DisplayCanvasUsedCreativesService displayCanvasUsedCreativesService) {
        this.displayCanvasUploadService = displayCanvasUploadService;
        this.displayCanvasAuthService = displayCanvasAuthService;
        this.displayCanvasCampaignService = displayCanvasCampaignService;
        this.displayCanvasUsedCreativesService = displayCanvasUsedCreativesService;
    }

    /**
     * @param operatorUid
     * @param clientId
     * @return Проверить курлом:
     * curl "http://12007.beta5.intapi.direct.yandex.ru/DisplayCanvas/authenticate/?client_id=450488&user_id=221776172"
     * curl "http://localhost:8090/DisplayCanvas/authenticate/?client_id=450488&user_id=221776172"
     */
    @GET
    @ApiOperation(
            value = "получение прав пользователя",
            nickname = "authenticate",
            httpMethod = "GET"
    )
    @ApiResponses(
            @ApiResponse(code = 400, message = "Не найден оператор или клиент",
                    response = ErrorResponse.class)
    )
    @RequestMapping(path = "authenticate",
            method = RequestMethod.GET,
            produces = MediaType.APPLICATION_JSON
    )
    public Privileges authenticate(
            @ApiParam(value = "текущий паспортный uid пользователя из конструктора креативов")
            @RequestParam(value = "user_id", required = false) @OperatorUid Long operatorUid,

            @ApiParam(value = "id клиента из баланса, на него хранятся креативы в конструкторе "
                    + "(передается как одноименный get параметр при открыти iframe c конструктором)"
            )
            @RequestParam(value = "client_id", required = false) @ClientIdParam Long clientId
    ) {
        return displayCanvasAuthService.authenticate(operatorUid, clientId);
    }


    /**
     * @return Проверить курлом:
     * curl "http://12007.beta5.intapi.direct.yandex.ru/DisplayCanvas/ping"
     * curl "http://localhost:8090/DisplayCanvas/ping"
     */
    @GET
    @ApiOperation(
            value = "проверка  работоспособности ручки authenticate",
            nickname = "ping",
            httpMethod = "GET"
    )
    @RequestMapping(path = "ping",
            method = RequestMethod.GET,
            produces = MediaType.APPLICATION_JSON
    )
    public String ping() {
        return "OK";
    }

    /**
     * @param operatorUid
     * @param clientId
     * @param creatives
     * @return
     */
    @POST
    @ApiOperation(
            value = "создание креативов",
            nickname = "upload_creatives",
            httpMethod = "POST"
    )
    @ApiResponses({
            @ApiResponse(code = 403, message = "Недостаточно прав для выполнения операции",
                    response = ErrorResponse.class),
            @ApiResponse(code = 400, message = "Некорректный запрос",
                    response = ErrorResponse.class)
    })
    @RequestMapping(path = "upload_creatives",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON,
            produces = MediaType.APPLICATION_JSON
    )
    @LoggingConfig(logRequestBody = EventType.ALL, logResponseBody = EventType.ALL)
    public CreativeUploadResponse uploadCreatives(
            @ApiParam(value = "текущий паспортный uid пользователя из конструктора креативов", required = true)
            @RequestParam(value = "operator_uid", required = false) @OperatorUid Long operatorUid,

            @ApiParam(value = "id клиента из баланса, на него хранятся креативы в конструкторе "
                    + "(передается как одноименный get параметр при открыти iframe c конструктором)",
                    required = true
            )
            @RequestParam(value = "client_id") @ClientIdParam long clientId,

            @ApiParam(value = "Список креативов", required = true)
            @RequestBody() List<CreativeUploadData> creatives
    ) {
        // метод принимает operatorUid - удобно для логирование, но параметр не обязателен, т.к. права не проверяем
        return displayCanvasUploadService.uploadCreatives(ClientId.fromLong(clientId), creatives);
    }

    /**
     * @param operatorUid
     * @param clientId
     * @param request
     * @return
     */
    @POST
    @ApiOperation(
            value = "получение кампаний по заданным креативам",
            nickname = "get_creatives_campaigns",
            httpMethod = "POST"
    )
    @ApiResponses({
            @ApiResponse(code = 400, message = "Некорректный запрос", response = ErrorResponse.class),
            @ApiResponse(code = 403, message = "Недостаточно прав для выполнения операции",
                    response = ErrorResponse.class),
            @ApiResponse(code = 200, message = "Ok", response = Object.class)
            //На самом деле возвращается мапа, но кодогенератор для тестов создает не компилируемый код
    })
    @RequestMapping(path = "get_creatives_campaigns",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON,
            produces = MediaType.APPLICATION_JSON
    )
    public Map<Long, List<CreativeCampaignResult>> getCreativesCampaigns(
            @ApiParam(value = "текущий паспортный uid пользователя из конструктора креативов", required = true)
            @RequestParam(value = "operator_uid") @OperatorUid long operatorUid,

            @ApiParam(value = "id клиента из баланса, на него хранятся креативы в конструкторе "
                    + "(передается как одноименный get параметр при открыти iframe c конструктором)",
                    required = true
            )
            @RequestParam(value = "client_id") @ClientIdParam long clientId,

            @ApiParam(value = "Список id креативов, по которым хотим найти кампании", required = true)
            @RequestBody() CreativeCampaignRequest request
    ) {
        return displayCanvasCampaignService.getCreativesCampaigns(operatorUid, ClientId.fromLong(clientId), request);
    }

    /**
     * @param clientId
     * @param creativeType
     * @param nextPageToken
     * @param limit
     * @param sort
     * @return
     */
    @GET
    @ApiOperation(
            value = "получение используемых клиентом креативов",
            nickname = "get_used_creatives",
            httpMethod = "GET"
    )
    @ApiResponses({
            @ApiResponse(code = 400, message = "Некорректный запрос", response = ErrorResponse.class),
            @ApiResponse(code = 200, message = "Ok", response = GetUsedCreativesResponse.class)
    })
    @RequestMapping(path = "get_used_creatives",
            method = RequestMethod.GET,
            produces = MediaType.APPLICATION_JSON
    )
    public GetUsedCreativesResponse getUsedCreatives(
            @ApiParam(value = "id клиента из баланса, на него хранятся креативы в конструкторе",
                    required = true
            )
            @RequestParam(value = "client_id") long clientId,

            @ApiParam(value = "тип креатива",
                    required = true
            )
            @RequestParam(value = "creative_type") String creativeType,

            @ApiParam(value = "токен, позволяющий получить следующий блок данных")
            @RequestParam(value = "next_page_token", required = false) String nextPageToken,

            @ApiParam(value = "лимит")
            @RequestParam(value = "limit", required = false) Integer limit,

            @ApiParam(value = "сортировка asc/desc по ид креатива")
            @RequestParam(value = "sort", required = false) String sort
    ) {
        GetUsedCreativesRequest request = new GetUsedCreativesRequest().withClientId(clientId)
                .withCreativeType(GetUsedCreativesType.getEnumByValue(creativeType))
                .withNextPageToken(nextPageToken)
                .withLimit(limit)
                .withSort(sort != null ? GetUsedCreativesSort.valueOf(sort.toUpperCase()) : null);
        return displayCanvasUsedCreativesService.getUsedCreatives(request);
    }
}
