package ru.yandex.direct.intapi.entity.display.canvas.model;

import java.io.IOException;
import java.util.Collections;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;

public class Privileges {

    private Set<Permission> permissions;

    protected Privileges(Set<Permission> permissions) {
        this.permissions = Collections.unmodifiableSet(permissions);
    }

    public static class Builder {
        private Set<Permission> tmp;

        public Builder() {
            tmp = new HashSet<>();
        }

        public Privileges build() {
            return new Privileges(tmp);
        }

        public Privileges noPermissions() {
            return new Privileges(Collections.emptySet());
        }

        public Builder addPermission(Permission permission) {
            tmp.add(permission);
            return this;
        }
    }

    @JsonSerialize(using = LowerCaseSerializer.class)
    public enum Permission {
        /**
         * User permitted only to see creative preview
         */
        PREVIEW("Preview rights are required"),

        /**
         * User permitted to see creatives. Need for support crew
         */
        CREATIVE_GET("View creative rights are required"),

        /**
         * User permitted to create creatives
         */
        CREATIVE_CREATE("Write creative rights are required"),

        /**
         * User permitted to use idea facilities
         */
        IDEA("Idea rights are required"),

        /**
         * User permitted to use video addition facilities
         */
        VIDEO_ADDITION("Video addition rights are required"),

        /**
         * User permitted to use landings facilities
         */
        TURBO_LANDING("landing rights are required"),

        /**
         * User permitted to use desktop landing facilities
         */
        DESKTOP_LANDING("desktop landing rights are required"),

        /**
         * User permitted to use HTML5 facilities
         */
        HTML5_CREATIVES("HTML5 rights are required"),

        /**
         * User permitted to use adaptive creatives https://st.yandex-team.ru/DIRECT-82119
         */
        ADAPTIVE_CREATIVES("adaptive creatives rights are required"),

        /**
         * User is internal https://st.yandex-team.ru/DIRECT-111936
         */
        INTERNAL_USER("user must be internal");

        private String restrictedMsg;

        Permission(String restrictedMsg) {
            this.restrictedMsg = restrictedMsg;
        }

        public String getRestrictedMsg() {
            return restrictedMsg;
        }

        public static Optional<Permission> translateFromDirect(ActionType action) {
            switch (action) {
                case CREATIVE_GET:
                    return Optional.of(CREATIVE_GET);
                case CREATIVE_CREATE:
                    return Optional.of(CREATIVE_CREATE);
                default:
                    return Optional.empty();
            }
        }

        public static Optional<Permission> translateFromDirect(FeatureType feature) {
            switch (feature) {
                case IDEAS:
                    return Optional.of(IDEA);
                case VIDEO_ADDITION:
                    return Optional.of(VIDEO_ADDITION);
                case TURBO_LANDINGS:
                    return Optional.of(TURBO_LANDING);
                case HTML5_CREATIVES:
                    return Optional.of(HTML5_CREATIVES);
                case DESKTOP_LANDINGS:
                    return Optional.of(DESKTOP_LANDING);
                case ADAPTIVE_CREATIVES:
                    return Optional.of(ADAPTIVE_CREATIVES);
                case INTERNAL_USER:
                    return Optional.of(INTERNAL_USER);
                default:
                    return Optional.empty();
            }
        }
    }

    public boolean checkPermissionOn(Permission permission) {
        return permissions.contains(permission);
    }

    //to serialize actions and features in lower case
    public static class LowerCaseSerializer<T extends Enum> extends JsonSerializer<T> {
        @Override
        public void serialize(T value, JsonGenerator generator,
                              SerializerProvider provider) throws IOException {
            generator.writeString(value.name().toLowerCase());
        }
    }

    public Set<Permission> getPermissions() {
        return permissions;
    }
}
