package ru.yandex.direct.intapi.entity.display.canvas.service;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;

@Service
public class AuthCreativeValidationService {

    private final RbacService rbacService;
    private final ClientService clientService;
    private final ShardHelper shardHelper;

    @Autowired
    public AuthCreativeValidationService(RbacService rbacService,
                                         ClientService clientService, ShardHelper shardHelper) {
        this.rbacService = rbacService;
        this.clientService = clientService;
        this.shardHelper = shardHelper;
    }

    public long validateAndGetClientChiefUid(long operatorUid, ClientId clientId) throws IllegalArgumentException {
        long clientChiefUid;
        validateObjectExists(operatorUid, clientId);
        // если предыдущий вызов не выбросил IllegalArgumentException, clientId валидный
        clientChiefUid = rbacService.getChiefByClientId(clientId);
        validateClientRole(clientId, clientChiefUid);
        return clientChiefUid;
    }

    /**
     * Проверяется, что параметры запроса являются валидными, и оператор и клиент существуют
     *
     * @throws IllegalArgumentException если параметры невалидны
     */
    public void validateAuthParam(long operatorUid, ClientId clientId) throws IllegalArgumentException {
        Assert.isTrue(isValidId(operatorUid), "incorrect param operator_uid");
        Assert.isTrue(isValidId(clientId.asLong()), "incоrrect param client_id");
    }

    /**
     * Проверяется, что параметры запроса соответствуют существующим объектам
     *
     * @throws IllegalArgumentException если параметры указывают на отсутствующие объекты
     */
    private void validateObjectExists(long operatorUid, ClientId clientId) {
        Assert.isTrue(isUserExists(operatorUid), String.format("operator_uid %s not found", operatorUid));
        Assert.isTrue(isClientExists(clientId), String.format("client_id %s not found", clientId));
    }

    /**
     * @throws IllegalArgumentException если роль клиента отлична от {@link RbacRole#CLIENT}
     */
    private void validateClientRole(ClientId clientId, long clientChiefUid) throws IllegalArgumentException {
        Assert.isTrue(isClientHasClientRole(clientChiefUid),
                String.format("client with client_id %s has wrong role", clientId));
    }

    private boolean isClientExists(ClientId clientId) {
        return shardHelper.isExistentClientId(clientId.asLong());
    }

    private boolean isUserExists(long uid) {
        return shardHelper.isExistentUid(uid);
    }

    private boolean isValidId(long id) {
        return id > 0;
    }

    /**
     * Проверяет, что переданный {@code clientId} относится к кленту с ролью {@link RbacRole#CLIENT}
     *
     * @param clientChiefUid chiefUid клиента
     * @return {@code true} если роль клиента с указанным {@code clientId} &ndash; {@link RbacRole#CLIENT}
     */
    private boolean isClientHasClientRole(long clientChiefUid) {
        RbacRole clientRole = rbacService.getUidRole(clientChiefUid);
        return RbacRole.CLIENT == clientRole;
    }

}
