package ru.yandex.direct.intapi.entity.display.canvas.service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.creative.service.CreativeService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.IntApiException;
import ru.yandex.direct.intapi.entity.display.canvas.model.ActionType;
import ru.yandex.direct.intapi.entity.display.canvas.model.AuthResponse;
import ru.yandex.direct.intapi.entity.display.canvas.model.FeatureType;
import ru.yandex.direct.intapi.entity.display.canvas.model.Privileges;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;

import static ru.yandex.direct.feature.FeatureName.CANVAS_AVAILABLE_ADAPTIVE_CREATIVE;
import static ru.yandex.direct.feature.FeatureName.CPM_BANNER;
import static ru.yandex.direct.feature.FeatureName.CPM_DEALS;
import static ru.yandex.direct.feature.FeatureName.DESKTOP_LANDING;
import static ru.yandex.direct.intapi.entity.display.canvas.model.FeatureType.HTML5_CREATIVES;
import static ru.yandex.direct.intapi.entity.display.canvas.model.FeatureType.IDEAS;
import static ru.yandex.direct.intapi.entity.display.canvas.model.FeatureType.IMAGES;
import static ru.yandex.direct.intapi.entity.display.canvas.model.FeatureType.INTERNAL_USER;
import static ru.yandex.direct.intapi.entity.display.canvas.model.FeatureType.TURBO_LANDINGS;
import static ru.yandex.direct.intapi.entity.display.canvas.model.FeatureType.VIDEO_ADDITION;
import static ru.yandex.direct.intapi.entity.display.canvas.model.Privileges.Permission.CREATIVE_GET;
import static ru.yandex.direct.intapi.entity.display.canvas.model.Privileges.Permission.PREVIEW;


@Service
public class DisplayCanvasAuthService {
    private static final Logger logger = LoggerFactory.getLogger(DisplayCanvasAuthService.class);
    private static final List<ActionType> ALL_RIGHTS =
            Arrays.asList(
                    ActionType.CREATIVE_CREATE, ActionType.CREATIVE_GET, ActionType.CREATIVE_EDIT,
                    ActionType.CREATIVE_DELETE);

    private final RbacService rbacService;
    private final ClientService clientService;
    private final AuthCreativeValidationService authCreativeValidationService;
    private final CreativeService creativeService;
    private final FeatureService featureService;
    private final ShardHelper shardHelper;
    private static final Privileges GUEST = guestPrivileges();

    @Autowired
    public DisplayCanvasAuthService(RbacService rbacService,
                                    ClientService clientService,
                                    AuthCreativeValidationService authCreativeValidationService,
                                    CreativeService creativeService, FeatureService featureService,
                                    ShardHelper shardHelper) {
        this.rbacService = rbacService;
        this.clientService = clientService;
        this.authCreativeValidationService = authCreativeValidationService;
        this.creativeService = creativeService;
        this.featureService = featureService;
        this.shardHelper = shardHelper;
    }

    // для внешнего доступа следует использовать метод authenticate
    AuthResponse auth(long operatorUid, ClientId clientId) {
        logger.debug("process request auth({client_id='{}', operator_uid='{}'})", clientId, operatorUid);

        // возвращаем 400 на невалидных id, и пустой ответ на неизвестных
        try {
            authCreativeValidationService.validateAuthParam(operatorUid, clientId);
        } catch (IllegalArgumentException e) {
            throw new IntApiException(HttpStatus.BAD_REQUEST,
                    new ErrorResponse(ErrorResponse.ErrorCode.BAD_PARAM, e.getLocalizedMessage()));
        }

        List<ActionType> availableOperations;
        try {
            long clientChiefUid = authCreativeValidationService.validateAndGetClientChiefUid(operatorUid, clientId);
            long effectiveOperatorUid = rbacService.getEffectiveOperatorUid(operatorUid, clientChiefUid);
            RbacRole operatorRole = getOperatorRole(effectiveOperatorUid, clientChiefUid);
            availableOperations = getAvailableOperations(effectiveOperatorUid, clientId, clientChiefUid, operatorRole);
            List<FeatureType> availableFeatures =
                    availableOperations.isEmpty() // если у оператора нет прав на клиента..
                            ? Collections.emptyList() // ..то возвращаем пустой список фич
                            : getAvailableFeatures(effectiveOperatorUid, clientId, operatorRole);
            logger.debug("response for ({client_id='{}', operator_uid='{}'}): {}, {}", clientId, operatorUid,
                    availableOperations, availableFeatures);
            return new AuthResponse(availableOperations, availableFeatures);
        } catch (IllegalArgumentException e) {
            logger.debug("empty response for non-client ({client_id='{}', operator_uid='{}'})", clientId, operatorUid);
            return new AuthResponse(Collections.emptyList(), Collections.emptyList());
        }
    }

    private List<FeatureType> getAvailableFeatures(long operatorUid, ClientId clientId, RbacRole operatorRole) {
        List<FeatureType> features = new ArrayList<>(Collections.singletonList(IMAGES));
        features.add(VIDEO_ADDITION);
        features.add(TURBO_LANDINGS);

        if (featureService.isEnabledForClientId(clientId, DESKTOP_LANDING)) {
            features.add(FeatureType.DESKTOP_LANDINGS);
        }

        if (featureService.isEnabledForClientId(clientId, CANVAS_AVAILABLE_ADAPTIVE_CREATIVE)) {
            features.add(FeatureType.ADAPTIVE_CREATIVES);
        }

        long operatorClientId = shardHelper.getClientIdByUid(operatorUid);

        if (operatorRole == RbacRole.SUPER
                || creativeService.clientAlreadyHasHtml5Creatives(clientId)
                || featureService.isEnabledForClientId(clientId, CPM_BANNER)
                || featureService.isEnabledForClientId(ClientId.fromLong(operatorClientId), CPM_DEALS)) {
            features.add(HTML5_CREATIVES);
        }

        if (operatorRole.isInternal()) {
            features.add(INTERNAL_USER);
        }

        return features;
    }

    private RbacRole getOperatorRole(long operatorUid, long clientChiefUid) {
        // оптимизация: для случая, когда operatorUid == clientChiefUid, можем сократить количество обращений к RBAC
        // на этапе валидации уже проверили, что роль клиента с clientId -- RbacRole.CLIENT
        if (operatorUid == clientChiefUid) {
            return RbacRole.CLIENT;
        }

        RbacRole operatorRole = rbacService.getUidRole(operatorUid);
        logger.debug("Role by operator_uid {}: {}", operatorUid, operatorRole);
        return operatorRole;
    }

    private List<ActionType> getAvailableOperations(long operatorUid, ClientId clientId, long clientChiefUid,
                                                    RbacRole operatorRole) {
        boolean operatorIsClientChief = operatorUid == clientChiefUid;

        switch (operatorRole) {
            case SUPER:
                // SUPER может всё и со всеми
                return ALL_RIGHTS;
            case AGENCY:
            case MANAGER:
                // Агенства и Менеджеры могут всё с материалами своих клиентов
                if (rbacService.isOwner(operatorUid, clientChiefUid)) {
                    return ALL_RIGHTS;
                }
                break;
            case PLACER:
            case MEDIA:
            case SUPERREADER:
                // Placer'ы, Медиапланировщики и Суперчитатели могут просматривать материалы всех клиентов
                return Collections.singletonList(ActionType.CREATIVE_GET);
            case SUPPORT:
                // Поддержка может изменять, но не может удалять материалы
                return Arrays.asList(ActionType.CREATIVE_GET, ActionType.CREATIVE_CREATE, ActionType.CREATIVE_EDIT);
            case LIMITED_SUPPORT:
                // LIMITED_SUPPORT могут просматривать материалы своих клиентов
                if (rbacService.isOwner(operatorUid, clientChiefUid)) {
                    return Collections.singletonList(ActionType.CREATIVE_GET);
                }
                break;
            case CLIENT:
                // Клиент (как и его представитель) может изменять только свои материалы
                // оптимизация: не вычисляем operatorChiefUid, если operator == clientChiefUid
                if (!operatorIsClientChief) {
                    long operatorChiefUid = rbacService.getChief(operatorUid, operatorRole);
                    if (operatorChiefUid != clientChiefUid) {
                        logger.debug(
                                "Given operator {} is not related to clientId {}. " +
                                        "Expected operatorChiefUid {} is not equal to clientChiefUid {}",
                                operatorUid, clientId, operatorChiefUid, clientChiefUid
                        );
                        break;
                    }
                }
                // тут по факту наш operator -- это обычный клиент и мы должны лишь удостовериться,
                // что или у него нет агенства, или оно есть и дало ему права на редактирование
                // (проверяется через права пользователя и через право makeSimpleSubClient в rbac для агенства)
                if (rbacService.isUnderAgency(clientChiefUid)) {
                    boolean isSuperSubclient = clientService.isSuperSubclient(clientId);
                    if (!isSuperSubclient) {
                        // Клиент управляется агенством, и агенство не дало ему права на изменение своих материалов
                        return Collections.singletonList(ActionType.CREATIVE_GET);
                    }
                }
                return ALL_RIGHTS;
            default:
                // operator has no rights for given clientId
        }

        return Collections.emptyList();
    }

    public Privileges authenticate(Long userId, Long clientIdRaw) {

        if (userId == null || clientIdRaw == null) {
            return GUEST;
        }

        ClientId clientId = ClientId.fromLong(clientIdRaw);

        AuthResponse directPermissions = auth(userId, clientId);

        if (directPermissions.getAvailableFeatures().isEmpty() && directPermissions.getAvailableActions().isEmpty()) {
            return new Privileges.Builder().noPermissions();
        }

        Privileges.Builder builder = new Privileges.Builder();
        builder.addPermission(PREVIEW);

        directPermissions.getAvailableActions().stream()
                .map(action -> Privileges.Permission.translateFromDirect(action))
                .filter(Optional::isPresent)
                .map(Optional::get)
                .forEach(action -> builder.addPermission(action));

        Stream.concat(directPermissions.getAvailableFeatures().stream(), Stream.of(IMAGES, IDEAS))
                .map(action -> Privileges.Permission.translateFromDirect(action))
                .filter(Optional::isPresent)
                .map(Optional::get)
                .forEach(feature -> builder.addPermission(feature));

        Privileges privileges = builder.build();

        //if no creative get permission granted by direct clientId is considered as untrusted
        if (!privileges.checkPermissionOn(CREATIVE_GET)) {
            return GUEST;
        }

        return privileges;
    }

    private static Privileges guestPrivileges() {
        return new Privileges.Builder()
                .addPermission(PREVIEW)
                .build();
    }


}
