package ru.yandex.direct.intapi.entity.display.canvas.service;

import java.util.List;
import java.util.Map;

import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.CampaignName;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.IntApiException;
import ru.yandex.direct.intapi.entity.display.canvas.model.AuthResponse;
import ru.yandex.direct.intapi.entity.display.canvas.model.CreativeCampaignRequest;
import ru.yandex.direct.intapi.entity.display.canvas.model.CreativeCampaignResult;
import ru.yandex.direct.intapi.entity.display.canvas.validation.CreativeCampaignValidationService;
import ru.yandex.direct.intapi.validation.IntApiDefect;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.intapi.entity.display.canvas.model.ActionType.CREATIVE_GET;
import static ru.yandex.direct.intapi.validation.ValidationUtils.checkResult;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class DisplayCanvasCampaignService {

    private static final String BASE_CAMP_URL = "https://direct.%yandex_tld%/registered/main.pl?cmd=showCamp&cid=";
    private final ShardHelper shardHelper;
    private final RbacService rbacService;
    private final DisplayCanvasAuthService displayCanvasAuthService;
    private final CreativeCampaignValidationService creativeCampaignValidationService;
    private final CampaignRepository campaignRepository;

    @Autowired
    public DisplayCanvasCampaignService(ShardHelper shardHelper, RbacService rbacService,
                                        DisplayCanvasAuthService displayCanvasAuthService,
                                        CreativeCampaignValidationService creativeCampaignValidationService,
                                        CampaignRepository campaignRepository) {
        this.shardHelper = shardHelper;
        this.rbacService = rbacService;
        this.displayCanvasAuthService = displayCanvasAuthService;
        this.creativeCampaignValidationService = creativeCampaignValidationService;
        this.campaignRepository = campaignRepository;
    }

    public Map<Long, List<CreativeCampaignResult>> getCreativesCampaigns(long operatorUid, ClientId clientId,
                                                                         CreativeCampaignRequest request) {
        AuthResponse authResponse = displayCanvasAuthService.auth(operatorUid, clientId);
        if (!authResponse.getAvailableActions().contains(CREATIVE_GET)) {
            throw new IntApiException(HttpStatus.FORBIDDEN,
                    new ErrorResponse(ErrorResponse.ErrorCode.PERMISSION_DENIED, "Permission denied"));
        }
        ValidationResult<CreativeCampaignRequest, IntApiDefect> vr =
                creativeCampaignValidationService.validate(request);
        checkResult(vr);

        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        long clientChiefUid = rbacService.getChiefByClientId(clientId);
        String chiefLogin = shardHelper.getLoginByUid(clientChiefUid);
        Map<Long, List<CampaignName>> creativesCampaignNames = campaignRepository
                .getCampaignsByCreativeIds(shard, clientId, request.getCreativeIds());
        return EntryStream.of(creativesCampaignNames)
                .mapValues(lst -> mapList(lst, cn -> toCreativeCampaignResult(chiefLogin, cn)))
                .toMap();
    }

    private CreativeCampaignResult toCreativeCampaignResult(String chiefLogin, CampaignName campaignName) {
        return new CreativeCampaignResult()
                .withCampaignId(campaignName.getId())
                .withName(campaignName.getName())
                .withUrl(BASE_CAMP_URL + campaignName.getId() + "&ulogin=" + chiefLogin);
    }

}
