package ru.yandex.direct.intapi.entity.display.canvas.service;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.creative.model.ModerationInfo;
import ru.yandex.direct.core.entity.creative.model.SourceMediaType;
import ru.yandex.direct.core.entity.creative.model.StatusModerate;
import ru.yandex.direct.core.entity.creative.repository.CreativeMappings;
import ru.yandex.direct.core.entity.creative.service.CreativeService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.intapi.IntApiException;
import ru.yandex.direct.intapi.entity.display.canvas.model.CreativeUploadComposedFrom;
import ru.yandex.direct.intapi.entity.display.canvas.model.CreativeUploadData;
import ru.yandex.direct.intapi.entity.display.canvas.model.CreativeUploadResponse;
import ru.yandex.direct.intapi.entity.display.canvas.model.CreativeUploadResult;
import ru.yandex.direct.intapi.entity.display.canvas.model.CreativeUploadType;
import ru.yandex.direct.intapi.entity.display.canvas.validation.UploadCreativesValidationService;
import ru.yandex.direct.intapi.validation.IntApiDefect;
import ru.yandex.direct.intapi.validation.kernel.ValidationResultConversionService;
import ru.yandex.direct.intapi.validation.model.IntapiError;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.creative.model.StatusModerate.ADMINREJECT;
import static ru.yandex.direct.core.entity.creative.model.StatusModerate.YES;
import static ru.yandex.direct.core.entity.creative.service.CreativeService.HTML_5_MEDIA_TYPES_FOR_GENERATED_CREATIVE;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class DisplayCanvasUploadService {

    private final Logger logger = LoggerFactory.getLogger(DisplayCanvasUploadService.class);

    private final UploadCreativesValidationService uploadCreativesValidatorService;
    private final ValidationResultConversionService validationResultConversionService;
    private final CreativeService creativeService;

    @Autowired
    public DisplayCanvasUploadService(
            UploadCreativesValidationService uploadCreativesValidatorService,
            ValidationResultConversionService validationResultConversionService,
            CreativeService creativeService
    ) {
        this.uploadCreativesValidatorService = uploadCreativesValidatorService;
        this.validationResultConversionService = validationResultConversionService;
        this.creativeService = creativeService;
    }

    public CreativeUploadResponse uploadCreatives(ClientId clientId,
                                                  List<CreativeUploadData> creativeUploadData) {
        // авторазиация не нужна, заливает креативы сам канвас, формируя данные на своей стороне и после проверки прав
        // наличие проверки в этом месте усложняет миграции, приходится подставлять uid какого-нибудь супера
        ValidationResult<List<CreativeUploadData>, IntApiDefect> requestVR =
                uploadCreativesValidatorService.validate(creativeUploadData);
        if (requestVR.hasAnyErrors()) {
            throw new IntApiException(HttpStatus.BAD_REQUEST, "wrong request data");
        }

        List<Creative> creatives = mapList(creativeUploadData, cr -> toCreative(clientId, cr));

        List<Result<Long>> result = creativeService.createOrUpdate(creatives, clientId).getResult();

        List<CreativeUploadResult> uploadResults = new ArrayList<>(creatives.size());

        for (int idx = 0; idx < creatives.size(); idx++) {
            Long creativeId = creatives.get(idx).getId();
            Result<Long> resultItem = result.get(idx);
            if (resultItem.isSuccessful()) {
                uploadResults.add(CreativeUploadResult.ok(creativeId));
            } else {
                uploadResults.add(CreativeUploadResult.error(creativeId,
                        convertErrorsToString(resultItem.getValidationResult())));
            }
        }

        return new CreativeUploadResponse(uploadResults);
    }

    private Creative toCreative(ClientId clientId, CreativeUploadData uploadData) {
        final CreativeUploadType creativeType = uploadData.getCreativeType();
        if (CreativeUploadType.VIDEO_ADDITION.equals(creativeType)) {
            return toVideoType(clientId, uploadData);
        } else if (CreativeUploadType.HTML5_CREATIVE.equals(creativeType)) {
            return toHtml5Creative(clientId, uploadData);
        } else {
            return toCanvasCreative(clientId, uploadData);
        }
    }

    private Creative toVideoType(ClientId clientId, CreativeUploadData uploadData) {
        Creative creative = new Creative();
        throwBadRequestOnNullModerationInfo(uploadData);
        fillCreativeCommonFields(clientId, uploadData, creative);

        CreativeType type = CreativeMappings.convertVideoType(ifNotNull(uploadData.getPresetId(), Integer::longValue));

        creative.withType(type)
                .withModerationInfo(uploadData.getModerationInfo())
                .withDuration(ifNotNull(uploadData.getDuration(), Double::longValue))
                .withStockCreativeId(uploadData.getStockCreativeId())
                .withAdditionalData(uploadData.getAdditionalData())
                .withHasPackshot(nvl(uploadData.getHasPackshot(), false))
                .withIsBrandLift(nvl(uploadData.getIsBrandLift(), false))
                .withIsAdaptive(false);
        return creative;
    }

    private Creative toHtml5Creative(ClientId clientId, CreativeUploadData uploadData) {
        Creative creative = new Creative();
        final SourceMediaType sourceMediaType = html5ComposedFromToMediaType(uploadData.getComposedFrom());
        fillCreativeCommonFields(clientId, uploadData, creative);
        creative.withArchiveUrl(uploadData.getArchiveUrl())
                .withYabsData(uploadData.getYabsData())
                .withModerationInfo(uploadData.getModerationInfo())
                .withSourceMediaType(sourceMediaType)
                .withIsGenerated(HTML_5_MEDIA_TYPES_FOR_GENERATED_CREATIVE.contains(sourceMediaType))
                .withType(CreativeType.HTML5_CREATIVE)
                .withHasPackshot(false)
                .withExpandedPreviewUrl(uploadData.getExpandedPreviewUrl())
                .withDuration(ifNotNull(uploadData.getDuration(), Double::longValue))
                .withIsBrandLift(nvl(uploadData.getIsBrandLift(), false))
                .withIsAdaptive(false)
                .withAdditionalData(uploadData.getAdditionalData());
        return creative;
    }

    private SourceMediaType html5ComposedFromToMediaType(CreativeUploadComposedFrom composedFrom) {
        SourceMediaType mediaType;
        if (CreativeUploadComposedFrom.GIF.equals(composedFrom)) {
            mediaType = SourceMediaType.GIF;
        } else if (CreativeUploadComposedFrom.JPG.equals(composedFrom)) {
            mediaType = SourceMediaType.JPG;
        } else if (CreativeUploadComposedFrom.PNG.equals(composedFrom)) {
            mediaType = SourceMediaType.PNG;
        } else if (composedFrom == null) {
            mediaType = null;
        } else {
            logger.error("html5ComposedFromToMediaType unknown composed from value: " + composedFrom.getValue());
            mediaType = null;
        }
        return mediaType;
    }

    private void throwBadRequestOnNullModerationInfo(CreativeUploadData uploadData) {
        if (uploadData.getModerationInfo() == null) {
            throw new IntApiException(HttpStatus.BAD_REQUEST, "moderation info can't be empty");
        }
    }

    private Creative toCanvasCreative(ClientId clientId, CreativeUploadData uploadData) {
        Creative creative = new Creative();
        throwBadRequestOnNullModerationInfo(uploadData);
        fillCreativeCommonFields(clientId, uploadData, creative);

        CreativeType type = CreativeMappings.convertCanvasType(ifNotNull(uploadData.getPresetId(), Integer::longValue));

        creative.withType(type)
                .withModerationInfo(uploadData.getModerationInfo())
                .withStockCreativeId(uploadData.getStockCreativeId())
                .withYabsData(uploadData.getYabsData())
                .withHasPackshot(false)
                .withIsBrandLift(nvl(uploadData.getIsBrandLift(), false))
                //InBanner creatives have duration due to having a video block
                .withDuration(ifNotNull(uploadData.getDuration(), Double::longValue))
                .withIsAdaptive(uploadData.getIsAdaptive() == null ? false : uploadData.getIsAdaptive());
        return creative;
    }

    // общие для всех креативов поля
    private void fillCreativeCommonFields(ClientId clientId, CreativeUploadData uploadData, Creative creative) {
        StatusModerate statusModerate = Optional.ofNullable(uploadData.getModerationInfo())
                .map(ModerationInfo::getAdminRejectReason)
                .map(r -> ADMINREJECT)
                .orElse(YES);
        creative
                .withId(uploadData.getCreativeId())
                .withName(uploadData.getCreativeName())
                .withStatusModerate(statusModerate)
                .withPreviewUrl(uploadData.getPreviewUrl())
                .withLivePreviewUrl(uploadData.getLivePreviewUrl())
                .withWidth(ifNotNull(uploadData.getWidth(), Integer::longValue))
                .withHeight(ifNotNull(uploadData.getHeight(), Integer::longValue))
                .withLayoutId(ifNotNull(uploadData.getPresetId(), Integer::longValue))
                .withModerateTryCount(0L)
                .withClientId(clientId.asLong());

    }

    private String convertErrorsToString(ValidationResult<?, Defect> vr) {
        return StreamEx.of(validationResultConversionService.buildIntapiValidationResult(vr).getErrors())
                .map(IntapiError::getDescription)
                .joining(", ");
    }
}
