package ru.yandex.direct.intapi.entity.display.canvas.service;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import com.google.common.collect.Iterables;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.creative.repository.CreativeConstants;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.IntApiException;
import ru.yandex.direct.intapi.entity.display.canvas.model.GetUsedCreativesPageToken;
import ru.yandex.direct.intapi.entity.display.canvas.model.GetUsedCreativesRequest;
import ru.yandex.direct.intapi.entity.display.canvas.model.GetUsedCreativesResponse;
import ru.yandex.direct.intapi.entity.display.canvas.model.GetUsedCreativesSort;
import ru.yandex.direct.intapi.entity.display.canvas.model.GetUsedCreativesType;
import ru.yandex.direct.intapi.entity.display.canvas.validation.DisplayCanvasUsedCreativeValidationService;
import ru.yandex.direct.intapi.validation.kernel.ValidationResultConversionService;
import ru.yandex.direct.intapi.validation.model.IntapiError;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.singleton;
import static ru.yandex.direct.core.entity.creative.repository.CreativeConstants.GET_USED_CREATIVES_LIMIT;
import static ru.yandex.direct.multitype.entity.LimitOffset.limited;

@Service
public class DisplayCanvasUsedCreativesService {
    private final ShardHelper shardHelper;
    private final CreativeRepository creativeRepository;
    private final DisplayCanvasUsedCreativeValidationService usedCreativeValidationService;
    private final ValidationResultConversionService validationResultConversionService;

    @Autowired
    public DisplayCanvasUsedCreativesService(ShardHelper shardHelper,
                                             CreativeRepository creativeRepository,
                                             DisplayCanvasUsedCreativeValidationService usedCreativeValidationService,
                                             ValidationResultConversionService validationResultConversionService) {
        this.shardHelper = shardHelper;
        this.creativeRepository = creativeRepository;
        this.usedCreativeValidationService = usedCreativeValidationService;
        this.validationResultConversionService = validationResultConversionService;
    }

    public GetUsedCreativesResponse getUsedCreatives(GetUsedCreativesRequest request) {
        ValidationResult<GetUsedCreativesRequest, Defect> vr =
                usedCreativeValidationService.validate(request);
        if (vr.hasAnyErrors()) {
            throw new IntApiException(HttpStatus.BAD_REQUEST,
                    new ErrorResponse(ErrorResponse.ErrorCode.BAD_PARAM, convertErrorsToString(vr)));
        }

        ClientId clientId = ClientId.fromLong(request.getClientId());
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        GetUsedCreativesPageToken oldToken = getPageTokenFromString(request.getNextPageToken());

        Set<CreativeType> types = toCreativeTypes(request.getCreativeType());
        GetUsedCreativesSort sort =
                oldToken != null ? oldToken.getSort()
                        : Optional.ofNullable(request.getSort()).orElse(GetUsedCreativesSort.ASC);
        Long lastCreativeId = oldToken != null ? oldToken.getCreativeId() : 0L;
        int limit = request.getLimit() != null ? request.getLimit() : GET_USED_CREATIVES_LIMIT;

        List<Long> creativeIds = creativeRepository
                .getClientUsedCreativeIds(shard, clientId, types, lastCreativeId,
                        SqlUtils.SortOrder.valueOf(sort.name()), limited(limit));
        lastCreativeId = creativeIds.size() == limit ? Iterables.getLast(creativeIds, null) : null;

        return new GetUsedCreativesResponse(creativeIds, toNextPageToken(sort, oldToken, lastCreativeId));
    }

    private Set<CreativeType> toCreativeTypes(GetUsedCreativesType usedCreativesType) {
        if (GetUsedCreativesType.VIDEO_ADDITION.equals(usedCreativesType)) {
            return CreativeConstants.VIDEO_TYPES;
        } else {
            return singleton(CreativeType.CANVAS);
        }
    }

    private GetUsedCreativesPageToken toNextPageToken(GetUsedCreativesSort sort,
                                                      GetUsedCreativesPageToken oldTocken,
                                                      Long lastCreativeId) {
        if (lastCreativeId == null) {
            return null;
        }
        return oldTocken == null ? new GetUsedCreativesPageToken().withCreativeId(lastCreativeId).withSort(sort)
                : oldTocken.withCreativeId(lastCreativeId);
    }

    private GetUsedCreativesPageToken getPageTokenFromString(String pageToken) {
        if (pageToken == null || pageToken.isEmpty()) {
            return null;
        }
        List<String> tokenParams = Arrays.asList(pageToken.split("-"));
        return new GetUsedCreativesPageToken()
                .withCreativeId(Long.valueOf(tokenParams.get(1)))
                .withSort(GetUsedCreativesSort.valueOf(tokenParams.get(0)));

    }

    private String convertErrorsToString(ValidationResult<?, Defect> vr) {
        return StreamEx.of(validationResultConversionService.buildIntapiValidationResult(vr).getErrors())
                .map(IntapiError::getDescription)
                .joining(", ");
    }
}
