package ru.yandex.direct.intapi.entity.emailvalidation.controller;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.ws.rs.POST;
import javax.ws.rs.core.MediaType;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.validator.routines.EmailValidator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.tvm.AllowServices;

import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_TEST;

@RestController
@RequestMapping("emailvalidation")
@Api(value = "API для валидации email-адресов")
@AllowServices(production = {DIRECT_WEB_PROD}, testing = {DIRECT_WEB_TEST, DIRECT_DEVELOPER})
public class EmailValidationController {
    private static final Logger logger = LoggerFactory.getLogger(EmailValidationController.class);
    private static final EmailValidator EMAIL_VALIDATOR = EmailValidator.getInstance();

    /**
     * Валидация email-адресов
     *
     * @param emails список строк с email-адресами
     * @return словарь из email-адресов в boolean валидности
     */
    @POST
    @ApiOperation(
            value = "валидация email",
            nickname = "emailvalidation",
            httpMethod = "POST",
            responseContainer = "Map"
    )
    @PostMapping(
            consumes = MediaType.APPLICATION_JSON,
            produces = MediaType.APPLICATION_JSON
    )
    public Map<String, Boolean> validate(@ApiParam(value = "список email-адресов", required = true)
                                         @RequestBody @Nonnull List<String> emails) {
        Map<String, Boolean> answer = new HashMap<>();
        for (String email : emails) {
            if (answer.containsKey(email)) {
                continue;
            }
            if (EMAIL_VALIDATOR.isValid(email)) {
                logger.debug("Email {} is valid", email);
                answer.put(email, true);
            } else {
                logger.debug("Email {} is invalid", email);
                answer.put(email, false);
            }
        }

        return answer;
    }
}
