package ru.yandex.direct.intapi.entity.feature.controller;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import ru.yandex.direct.core.entity.feature.container.FeatureRequest;
import ru.yandex.direct.core.entity.feature.container.FeaturesWithExpBoxes;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.intapi.entity.feature.model.AbtInfoIntapi;
import ru.yandex.direct.intapi.entity.feature.model.AbtInfoRequest;
import ru.yandex.direct.intapi.entity.feature.model.AbtInfoResponse;
import ru.yandex.direct.intapi.entity.feature.model.FeatureAccessResponse;
import ru.yandex.direct.intapi.entity.feature.model.GetAccessToFeatureRequest;
import ru.yandex.direct.intapi.entity.feature.model.GetClientsWithExplicitFeatureResponse;
import ru.yandex.direct.intapi.entity.feature.service.GetAccessToFeatureService;
import ru.yandex.direct.intapi.validation.model.IntapiResponse;
import ru.yandex.direct.intapi.validation.model.IntapiValidationResponse;
import ru.yandex.direct.tvm.AllowServices;

import static ru.yandex.direct.tvm.TvmService.DIRECT_API_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_API_SANDBOX;
import static ru.yandex.direct.tvm.TvmService.DIRECT_API_SANDBOX_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_API_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_CANVAS_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_CANVAS_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.DIRECT_INTAPI_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_INTAPI_SANDBOX;
import static ru.yandex.direct.tvm.TvmService.DIRECT_INTAPI_SANDBOX_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_INTAPI_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_SCRIPTS_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_SCRIPTS_SANDBOX;
import static ru.yandex.direct.tvm.TvmService.DIRECT_SCRIPTS_SANDBOX_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_SCRIPTS_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_TEST;
import static ru.yandex.direct.tvm.TvmService.LPC_SWITCHCONSTRUCTOR_PROD;
import static ru.yandex.direct.tvm.TvmService.LPC_SWITCHCONSTRUCTOR_TEST;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Controller
@RequestMapping(value = "/feature_dev",
        produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Api(tags = "feature_dev")
@AllowServices(
        production = {DIRECT_WEB_PROD, DIRECT_SCRIPTS_PROD, DIRECT_INTAPI_PROD, DIRECT_API_PROD, DIRECT_CANVAS_PROD,
                LPC_SWITCHCONSTRUCTOR_PROD, DIRECT_WEB_TEST, DIRECT_SCRIPTS_TEST, DIRECT_DEVELOPER},
        testing = {DIRECT_WEB_TEST, DIRECT_SCRIPTS_TEST, DIRECT_INTAPI_TEST, DIRECT_API_TEST, DIRECT_CANVAS_TEST,
                LPC_SWITCHCONSTRUCTOR_TEST, DIRECT_DEVELOPER},
        sandbox = {DIRECT_INTAPI_SANDBOX, DIRECT_API_SANDBOX, DIRECT_SCRIPTS_SANDBOX},
        sandboxTesting = {DIRECT_INTAPI_SANDBOX_TEST, DIRECT_API_SANDBOX_TEST, DIRECT_SCRIPTS_SANDBOX_TEST,
                DIRECT_DEVELOPER}
)
public class FeatureDevController {
    private final GetAccessToFeatureService getAccessToFeatureService;
    private final FeatureService featureService;

    @Autowired
    public FeatureDevController(GetAccessToFeatureService getAccessToFeatureService, FeatureService featureService) {
        this.getAccessToFeatureService = getAccessToFeatureService;
        this.featureService = featureService;
    }

    @ApiOperation(
            value = "get status of features for direct uids",
            httpMethod = "POST",
            nickname = "getStatusOfFeaturesForDirectUids"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = IntapiResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = IntapiResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = FeatureAccessResponse.class)
            }
    )
    @RequestMapping(path = "/access", method = RequestMethod.POST)
    @ResponseBody
    public IntapiResponse isFeatureEnabled(@RequestBody GetAccessToFeatureRequest getAccessToFeatureRequest) {
        return getAccessToFeatureService
                .isFeatureEnabled(getAccessToFeatureRequest.getUids(),
                        listToSet(getAccessToFeatureRequest.getClientIds(), ClientId::fromLong));
    }

    @RequestMapping(path = "/abt_info", method = RequestMethod.POST)
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = IntapiResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = IntapiResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = AbtInfoResponse.class)
            }
    )
    @ResponseBody
    public IntapiResponse getAbtInfo(@RequestBody AbtInfoRequest abtInfoRequest) {
        var featureRequest = new FeatureRequest()
                .withClientId(ClientId.fromLong(abtInfoRequest.getClientId()))
                .withHost(abtInfoRequest.getHost())
                .withIp(abtInfoRequest.getIp())
                .withUserAgent(abtInfoRequest.getUserAgent())
                .withYandexuid(abtInfoRequest.getYandexUid())
                .withYexpCookie(abtInfoRequest.getYexp())
                .withNpFixedFeatures(abtInfoRequest.getNpFixedFeatures())
                .withInterfaceLang(abtInfoRequest.getInterfaceLang());
        FeaturesWithExpBoxes featuresWithExpBoxes;
        var wantedFeatures = abtInfoRequest.getWantedFeatures();
        if (wantedFeatures == null) {
            featuresWithExpBoxes = featureService.getFeaturesWithExpBoxes(featureRequest);
        } else {
            featuresWithExpBoxes = featureService.getFeaturesWithExpBoxes(featureRequest, wantedFeatures);
        }

        return new AbtInfoResponse()
                .withResult(AbtInfoIntapi.fromFeaturesWithExpBoxes(featuresWithExpBoxes));
    }

    @ApiOperation(
            value = "возвращает список клиентов, которым явно включили указанную фичу",
            httpMethod = "GET"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad request", response = IntapiValidationResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = GetClientsWithExplicitFeatureResponse.class)
            }
    )
    @RequestMapping(path = "/get_clients_with_explicit_feature", method = RequestMethod.GET)
    @ResponseBody
    public IntapiResponse getClientsWithExplicitFeature(@RequestParam("feature_name") String featureName) {
        return getAccessToFeatureService.getClientsWithExplicitFeature(featureName);
    }
}
