package ru.yandex.direct.intapi.entity.feature.controller

import io.swagger.annotations.Api
import io.swagger.annotations.ApiOperation
import org.springframework.http.MediaType
import org.springframework.stereotype.Controller
import org.springframework.web.bind.annotation.GetMapping
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.RequestParam
import org.springframework.web.bind.annotation.ResponseBody
import ru.yandex.direct.core.entity.feature.model.FeatureHistory
import ru.yandex.direct.core.entity.feature.repository.FeatureHistoryRepository
import ru.yandex.direct.feature.FeatureHistoryEvent
import ru.yandex.direct.tvm.AllowServices
import ru.yandex.direct.tvm.TvmService

@Controller
@Api(tags = ["features_history"])
@RequestMapping(value = ["/features_history"])
@AllowServices(
    production = [
        TvmService.DIRECT_SCRIPTS_TEST,
        TvmService.DIRECT_DEVELOPER,
    ],
    testing = [
        TvmService.DIRECT_SCRIPTS_TEST,
        TvmService.DIRECT_DEVELOPER,
    ],
)
class FeatureHistoryController(
    private val featureHistoryRepository: FeatureHistoryRepository,
) {
    @ApiOperation(
        value = "Get features history",
        httpMethod = "GET",
        nickname = "features_history",
    )
    @GetMapping(produces = [MediaType.APPLICATION_JSON_UTF8_VALUE])
    @ResponseBody
    fun featuresHistory(@RequestParam("after_event_id") afterEventId: Long): List<FeatureHistoryEvent> {
        val featureHistory: Map<Long, List<FeatureHistory>> =
            featureHistoryRepository.getFeatureHistorySince(afterEventId)

        val earliestEvents: Map<Long, FeatureHistory> = featureHistory
            .mapValues { (_, events) -> events.minByOrNull { it.id }!! }
        val previousSettings: Map<Long, FeatureHistory> =
            featureHistoryRepository.getPrevFeatureSettings(earliestEvents)

        val eventsByFeature: Map<Long, List<FeatureHistoryEvent>> = featureHistory
            .mapValues { (featureId, events) ->
                // Хотим для каждого события найти предыдущее,
                // для этого добавляем в начало previousSettings[featureId] (возможно null)
                (listOf(previousSettings[featureId]) + events)
                    .zipWithNext()
                    .map { (prev, item) -> prev to item!! }
                    .map { (prev: FeatureHistory?, item: FeatureHistory) ->
                        FeatureHistoryEvent(
                            eventId = item.id,
                            featureId = item.featureId,
                            featureTextId = item.featureTextId.ifEmpty { null },
                            owner = item.newSettings.originalOwner
                                ?: prev?.newSettings?.originalOwner,
                            percentBefore = prev?.newSettings?.percent,
                            percentAfter = item.newSettings.percent,
                        )
                    }
            }

        return eventsByFeature.values.flatten()
            .sortedBy { it.eventId }
    }
}
