package ru.yandex.direct.intapi.entity.feature.service;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.feature.container.ChiefRepresentativeWithClientFeature;
import ru.yandex.direct.core.entity.feature.model.ClientFeature;
import ru.yandex.direct.core.entity.feature.model.FeatureState;
import ru.yandex.direct.core.entity.feature.service.FeatureManagingService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.intapi.IntApiException;
import ru.yandex.direct.intapi.entity.feature.model.ClientUidFeaturesIntapi;
import ru.yandex.direct.intapi.entity.feature.model.FeatureAccessResponse;
import ru.yandex.direct.intapi.entity.feature.model.GetClientsWithExplicitFeatureResponse;
import ru.yandex.direct.intapi.validation.kernel.ValidationResultConversionService;
import ru.yandex.direct.intapi.validation.model.IntapiResponse;
import ru.yandex.direct.result.Result;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;

@Service
public class GetAccessToFeatureService {
    private final FeatureService featureService;
    private final ValidationResultConversionService validationResultConversionService;
    private final FeatureManagingService featureManagingService;

    @Autowired
    public GetAccessToFeatureService(
            FeatureService featureService,
            ValidationResultConversionService validationResultConversionService,
            FeatureManagingService featureManagingService
    ) {
        this.featureService = featureService;
        this.validationResultConversionService = validationResultConversionService;
        this.featureManagingService = featureManagingService;
    }

    public IntapiResponse isFeatureEnabled(List<Long> uids, Set<ClientId> clientIds) {
        var clientUidFeaturesIntapi = new ClientUidFeaturesIntapi();
        if (clientIds != null) {
            var clientIdsFeatures =
                    EntryStream.of(featureService.getEnabled(clientIds))
                            .filterValues(features -> !features.isEmpty())
                            .mapValues(List::copyOf).toMap();
            clientUidFeaturesIntapi.withClientIdsFeatures(clientIdsFeatures);
        }

        if (uids != null) {
            var uidsFeatures = EntryStream.of(featureService.getEnabledForUids(uids)).mapValues(List::copyOf).toMap();
            clientUidFeaturesIntapi.withUidsFeatures(uidsFeatures);
        }

        return new FeatureAccessResponse()
                .withResult(clientUidFeaturesIntapi);
    }

    /**
     * Вычисляет список клиентов, которым явно выставили фичу featureName.
     * Может кинуть исключение, если имя фичи не валидное. В этом случае контроллер ответит 400-м кодом.
     */
    public GetClientsWithExplicitFeatureResponse getClientsWithExplicitFeature(String featureName) {
        Result<Map<Long, List<ChiefRepresentativeWithClientFeature>>> operationResult =
                featureManagingService.getFeaturesClients(
                        Collections.singletonList(featureName), FeatureState.ENABLED
                );
        if (hasValidationIssues(operationResult)) {
            throw new IntApiException(
                    HttpStatus.BAD_REQUEST,
                    validationResultConversionService.buildValidationResponse(operationResult)
            );
        }
        List<Long> clientIdList = operationResult.getResult().values().stream().flatMap(Collection::stream)
                .map(ChiefRepresentativeWithClientFeature::getClientFeature)
                .map(ClientFeature::getClientId)
                .map(ClientId::asLong)
                // потенциально из сервиса может прийти несколько строк с одним clientId
                .distinct()
                // сортировка для простоты тестирования
                .sorted(Comparator.naturalOrder())
                .collect(toList());
        return new GetClientsWithExplicitFeatureResponse().withResult(clientIdList);
    }
}
