package ru.yandex.direct.intapi.entity.geoproduct;

import java.util.Collections;
import java.util.Map;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.intapi.entity.image.model.BannerImageUploadResponse;
import ru.yandex.direct.intapi.entity.image.service.BannerImageService;
import ru.yandex.direct.intapi.validation.model.IntapiResponse;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.tvm.AllowServices;
import ru.yandex.direct.web.core.model.WebErrorResponse;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.intapi.entity.geoproduct.service.GeoProductTransferMoneyValidationService.DEFAULT_GEOPRODUCT_INTAPI_MONEY_TRANSFER_OPERATORS;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.DIRECT_INTAPI_TEST;
import static ru.yandex.direct.tvm.TvmService.GEO_PRODUCT_PROD;
import static ru.yandex.direct.tvm.TvmService.GEO_PRODUCT_TEST;

@RestController
@RequestMapping("geoproduct/image")
@Api(tags = "geoproduct_image", value = "загрузка картинок для нужд геопродукта")
@AllowServices(
        production = {GEO_PRODUCT_PROD},
        testing = {GEO_PRODUCT_TEST, DIRECT_DEVELOPER, DIRECT_INTAPI_TEST})
public class GeoProductImageController {

    private final BannerImageService bannerImageService;
    private final RbacService rbacService;

    @Autowired
    public GeoProductImageController(BannerImageService bannerImageService,
            RbacService rbacService) {
        this.bannerImageService = bannerImageService;
        this.rbacService = rbacService;
    }

    @RequestMapping(path = "upload_by_url", method = RequestMethod.POST, consumes =
            MediaType.APPLICATION_JSON_UTF8_VALUE)
    @ResponseBody
    @ApiOperation(value = "uploadImageByUrlForGeoProduct", httpMethod = "POST",
            nickname = "uploadImageByUrlForGeoProduct")
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = BannerImageUploadResponse.class)
            }
    )
    public IntapiResponse uploadImageByUrlForImageAd(
            @ApiParam(value = "url", required = true) @RequestParam("url") String url,
            @ApiParam(value = "image_type", required = true) @RequestParam("image_type") String imageType,
            @ApiParam(value = "client_id", required = true) @RequestParam("client_id") long clientId,
            @ApiParam(value = "operator_uid", required = true) @RequestParam("operator_uid") long operatorUid) {

        checkState(DEFAULT_GEOPRODUCT_INTAPI_MONEY_TRANSFER_OPERATORS.contains(operatorUid), "Operator ID not allowed");
        ClientId client = ClientId.fromLong(clientId);
        Map<ClientId, Long> chiefsByClientIds = rbacService.getChiefsByClientIds(Collections.singleton(client));
        checkState(!chiefsByClientIds.isEmpty(), "Client ID not found");
        boolean operatorCanEditCampaigns = rbacService.canWrite(operatorUid, chiefsByClientIds.get(client));
        checkState(operatorCanEditCampaigns, "Operator cannot edit client's campaigns");

        return bannerImageService.saveImage(url, imageType, client);
    }

}
