package ru.yandex.direct.intapi.entity.geoproduct.service;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import ru.yandex.direct.balance.client.exception.BalanceClientException;
import ru.yandex.direct.balance.client.model.request.createtransfermultiple.CreateTransferMultipleRequest;
import ru.yandex.direct.balance.client.model.request.createtransfermultiple.TransferTargetFrom;
import ru.yandex.direct.balance.client.model.request.createtransfermultiple.TransferTargetTo;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.WalletCampaign;
import ru.yandex.direct.core.entity.campaign.model.WalletTypedCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.model.ClientNds;
import ru.yandex.direct.core.entity.client.service.ClientNdsService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.service.integration.balance.BalanceService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.currency.Percent;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.IntApiException;
import ru.yandex.direct.intapi.entity.geoproduct.model.AccountTransferAmount;
import ru.yandex.direct.intapi.entity.geoproduct.model.TransferMoneyRequest;
import ru.yandex.direct.intapi.entity.geoproduct.model.TransferMoneyResponse;
import ru.yandex.direct.intapi.entity.geoproduct.model.WalletRestMoneyItem;
import ru.yandex.direct.utils.CollectionUtils;

import static com.google.common.base.Preconditions.checkState;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class GeoProductTransferMoneyService {

    private static final Logger logger = LoggerFactory.getLogger(GeoProductTransferMoneyService.class);

    private final BalanceService balanceService;
    private final ShardHelper shardHelper;

    private final int directServiceId;
    private final ClientNdsService clientNdsService;
    private final ClientService clientService;
    private final CampaignService campaignService;
    private final CampaignTypedRepository campaignTypedRepository;

    @Autowired
    public GeoProductTransferMoneyService(BalanceService balanceService,
            @Value("${balance.directServiceId}") int directServiceId,
            ShardHelper shardHelper,
            ClientNdsService clientNdsService, ClientService clientService,
            CampaignService campaignService,
            CampaignTypedRepository campaignTypedRepository) {
        this.balanceService = balanceService;
        this.directServiceId = directServiceId;
        this.shardHelper = shardHelper;
        this.clientNdsService = clientNdsService;
        this.clientService = clientService;
        this.campaignService = campaignService;
        this.campaignTypedRepository = campaignTypedRepository;
    }

    public List<WalletRestMoneyItem> getWalletsRestMoney(List<Long> clientIdList, List<Long> walletIdList) {

        if (!CollectionUtils.isEmpty(clientIdList)) {
            walletIdList = shardHelper.groupByShard(clientIdList, ShardKey.CLIENT_ID).stream()
                    .flatMapValues(List::stream)
                    .mapKeyValue((shard, clientId) -> campaignTypedRepository
                            .getClientsTypedCampaignsByType(shard, ClientId.fromLong(clientId),
                                    Collections.singleton(CampaignType.WALLET)))
                    .flatMapToEntry(Function.identity())
                    .values()
                    .map(WalletTypedCampaign.class::cast)
                    .map(WalletTypedCampaign::getId)
                    .toList();
        }

        Map<Long, ClientId> walletIdToClientIdMap = shardHelper.groupByShard(walletIdList, ShardKey.CID).stream()
                .mapKeyValue(campaignService::getWalletsByWalletCampaignIds)
                .flatMap(Collection::stream)
                .collect(toMap(WalletCampaign::getId, w -> ClientId.fromLong(w.getClientId())));

        Map<ClientId, Client> clientIdToClientMap =
                clientService.massGetClientsByClientIds(walletIdToClientIdMap.values());
        Collection<Client> clients = clientIdToClientMap.values();
        Map<Long, ClientNds> clientsNds =
                listToMap(clientNdsService.massGetEffectiveClientNds(clients), ClientNds::getClientId);

        List<WalletRestMoneyItem> walletRestMoneyItems = shardHelper.groupByShard(walletIdList, ShardKey.CID)
                .stream()
                .mapKeyValue(campaignService::getWalletsRestMoneyByWalletCampaignIds)
                .flatMapToEntry(Function.identity())
                .values()
                .map(walletRestMoney -> {
                    Long walletId = walletRestMoney.getWalletId();
                    ClientId clientId = walletIdToClientIdMap.get(walletId);
                    Client client = clientIdToClientMap.get(clientId);
                    Percent nds = clientsNds.get(clientId.asLong()).getNds();
                    BigDecimal walletRestMoneyWithoutNds = walletRestMoney.getRest().subtractNds(nds).bigDecimalValue();
                    BigDecimal walletRestMoneyWithNds = walletRestMoney.getRest().bigDecimalValue();

                    CurrencyCode currency = client.getWorkCurrency();
                    BigDecimal cashBackBonus = nvl(client.getCashBackBonus(), BigDecimal.ZERO);
                    BigDecimal cashBackBonusWithoutNds =
                            Money.valueOf(cashBackBonus, currency).subtractNds(nds).bigDecimalValue();
                    BigDecimal cashBackAwaitingBonus = nvl(client.getCashBackAwaitingBonus(), BigDecimal.ZERO);
                    BigDecimal cashBackAwaitingBonusWithoutNds =
                            Money.valueOf(cashBackAwaitingBonus, currency).subtractNds(nds).bigDecimalValue();

                    return new WalletRestMoneyItem()
                            .withClientId(clientId.asLong())
                            .withAccountId(walletId)
                            .withTotalCashback(cashBackBonus)
                            .withTotalCashbackWithoutNds(cashBackBonusWithoutNds)
                            .withAwaitingCashback(cashBackAwaitingBonus)
                            .withAwaitingCashbackWithoutNds(cashBackAwaitingBonusWithoutNds)
                            .withAmount(walletRestMoneyWithoutNds)
                            .withAmountWithNds(walletRestMoneyWithNds);
                })
                .collect(Collectors.toList());


        return walletRestMoneyItems;
    }

    public TransferMoneyResponse transferMoney(TransferMoneyRequest transferMoneyRequest) {
        CreateTransferMultipleRequest balanceRequest = convert(transferMoneyRequest);

        try {
            balanceService.createTransferMultiple(balanceRequest);
            return new TransferMoneyResponse(true);
        } catch (BalanceClientException balanceException) {
            logger.error(
                    String.format("Failed to transfer money from accountId: %s to accounts: %s for operator: %s",
                            transferMoneyRequest.getFromAccountId(),
                            transferMoneyRequest.getToAccounts(),
                            transferMoneyRequest.getOperatorUid()),
                    balanceException);
            throw new IntApiException(HttpStatus.INTERNAL_SERVER_ERROR,
                    new ErrorResponse(ErrorResponse.ErrorCode.INTERNAL_ERROR, "BalanceException"));
        }
    }

    public CreateTransferMultipleRequest convert(TransferMoneyRequest transferMoneyRequest) {
        Long operatorUid = transferMoneyRequest.getOperatorUid();
        checkState(operatorUid != null && transferMoneyRequest.getFromAccountId() != null
                && !CollectionUtils.isEmpty(transferMoneyRequest.getToAccounts()));

        Map<Long, BigDecimal> walletIdToTransferAmounts = listToMap(transferMoneyRequest.getToAccounts(),
                AccountTransferAmount::getAccountId, AccountTransferAmount::getAmount);

        Set<Long> walletIdsTo = walletIdToTransferAmounts.keySet();
        checkState(!CollectionUtils.isEmpty(walletIdsTo));

        Long walletIdFrom = transferMoneyRequest.getFromAccountId();
        List<Long> allWalletIds = StreamEx.of(walletIdsTo)
                .append(walletIdFrom)
                .toList();

        Map<Long, WalletCampaign> walletsByIds = getWalletsByIds(allWalletIds);
        Map<Long, ClientNds> clientNdsMap = getClientNdsMap(walletsByIds);

        Map<Long, BigDecimal> walletIdToTransferAmountWithNds =
                addNds(walletIdToTransferAmounts, walletsByIds, clientNdsMap);

        BigDecimal walletsToSum = walletIdToTransferAmountWithNds.values().stream()
                .reduce(BigDecimal::add).orElse(BigDecimal.ZERO);

        WalletCampaign walletCampaignFrom = walletsByIds.get(walletIdFrom);
        BigDecimal walletFromSumForBalance = getWalletSumForBalance(walletCampaignFrom, walletsToSum);
        BigDecimal walletFromSumForBalanceAfter = walletFromSumForBalance.subtract(walletsToSum);

        return new CreateTransferMultipleRequest()
                .withOperatorUid(operatorUid)
                .withTransferTargetsFrom(List.of(
                        new TransferTargetFrom()
                                .withServiceId(directServiceId)
                                .withCampaignId(walletCampaignFrom.getId())
                                .withQtyOld(walletFromSumForBalance)
                                .withQtyNew(walletFromSumForBalanceAfter)))
                .withTransferTargetsTo(mapList(walletIdsTo, walletIdTo ->
                        new TransferTargetTo()
                                .withServiceId(directServiceId)
                                .withCampaignId(walletIdTo)
                                .withQtyDelta(walletIdToTransferAmountWithNds.get(walletIdTo))));
    }

    private BigDecimal getWalletSumForBalance(WalletCampaign walletCampaignFrom, BigDecimal walletsToSum) {
        Long walletIdFrom = walletCampaignFrom.getId();
        int shard = shardHelper.getShardByCampaignId(walletIdFrom);

        WalletTypedCampaign walletTypedCampaign = (WalletTypedCampaign) campaignTypedRepository
                .getTypedCampaigns(shard, Collections.singleton(walletIdFrom)).get(0);

        BigDecimal walletFromRestMoney = campaignService
                .getWalletsRestMoneyByWalletCampaignIds(shard, Collections.singleton(walletIdFrom))
                .get(walletIdFrom)
                .getRest()
                .bigDecimalValue();
        checkState(walletFromRestMoney.compareTo(walletsToSum) > 0, "Not enough sum on source account");

        BigDecimal walletSumForBalance;
        //https://a.yandex-team.ru/arc/commit/r4923332#file-/migration/svn/direct/trunk/protected/MoneyTransfer.pm:R1110
        if (walletTypedCampaign.getIsSumAggregated()) {
            walletSumForBalance = walletCampaignFrom.getSumBalance();
        } else {
            walletSumForBalance = walletCampaignFrom.getSum();
        }

        return walletSumForBalance;
    }

    private Map<Long, BigDecimal> addNds(Map<Long, BigDecimal> walletIdToTransferAmounts,
            Map<Long, WalletCampaign> walletsByIds, Map<Long, ClientNds> clientNdsMap) {
        return EntryStream.of(walletIdToTransferAmounts)
                .mapToValue((walletId, amount) -> {
                    WalletCampaign wallet = walletsByIds.get(walletId);
                    ClientNds clientNds = clientNdsMap.get(wallet.getClientId());
                    return Money.valueOf(amount, wallet.getCurrency()).addNds(clientNds.getNds()).bigDecimalValue();
                })
                .toMap();
    }

    @Nullable
    private Map<Long, ClientNds> getClientNdsMap(Map<Long, WalletCampaign> walletsByIds) {
        Set<ClientId> allClientIds = Set.copyOf(mapList(walletsByIds.values(),
                w -> ClientId.fromLong(w.getClientId())));
        Collection<Client> allClients = clientService.massGetClientsByClientIds(allClientIds).values();
        return listToMap(clientNdsService.massGetEffectiveClientNds(allClients), ClientNds::getClientId);
    }

    private Map<Long, WalletCampaign> getWalletsByIds(List<Long> allWalletIds) {
        List<WalletCampaign> allWallets = shardHelper.groupByShard(allWalletIds, ShardKey.CID).stream()
                .mapKeyValue(campaignService::getWalletsByWalletCampaignIds)
                .flatMap(Collection::stream)
                .collect(Collectors.toList());
        return listToMap(allWallets, WalletCampaign::getId);
    }

}
