package ru.yandex.direct.intapi.entity.geoproduct.service;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.intapi.entity.geoproduct.model.AccountTransferAmount;
import ru.yandex.direct.intapi.entity.geoproduct.model.TransferMoneyRequest;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static java.util.Collections.singleton;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notInSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.NumberConstraints.greaterThan;

@Service
public class GeoProductTransferMoneyValidationService {

    public static final Set<Long> DEFAULT_GEOPRODUCT_INTAPI_MONEY_TRANSFER_OPERATORS =
            ImmutableSet.of(745643215L, 745641711L, 745642534L, 745644061L);
    private final RbacService rbacService;

    @Autowired
    public GeoProductTransferMoneyValidationService(RbacService rbacService) {
        this.rbacService = rbacService;
    }

    public ValidationResult<TransferMoneyRequest, Defect> validate(TransferMoneyRequest request) {
        ItemValidationBuilder<TransferMoneyRequest, Defect> ivb = ItemValidationBuilder.of(request);

        Map<Long, BigDecimal> ordersTo = listToMap(nvl(request.getToAccounts(), emptyList()),
                AccountTransferAmount::getAccountId, AccountTransferAmount::getAmount);

        Long operatorUid = request.getOperatorUid();
        Long orderIdFrom = request.getFromAccountId();
        Set<Long> orderIdsTo = ordersTo.keySet();

        Collection<Long> allCampaignIds = Sets.union(orderIdsTo, singleton(orderIdFrom));
        Set<Long> writableCampaignIds = rbacService.getWritableCampaigns(operatorUid, allCampaignIds);

        Set<Long> validOperatorUids = DEFAULT_GEOPRODUCT_INTAPI_MONEY_TRANSFER_OPERATORS;

        ivb.item(operatorUid, "operatorUid")
                .check(notNull())
                .check(validId())
                .check(inSet(validOperatorUids));
        ivb.item(orderIdFrom, "fromAccountId")
                .check(notNull())
                .check(validId())
                .check(inSet(writableCampaignIds))
                .check(notInSet(orderIdsTo));
        ivb.list(List.copyOf(request.getToAccounts()), "toAccounts")
                .check(notNull())
                .checkEach(notNull())
                .check(notEmptyCollection())
                .checkEachBy(v -> validateAccountTransferAmount(v, writableCampaignIds));

        return ivb.getResult();
    }

    private ValidationResult<AccountTransferAmount, Defect> validateAccountTransferAmount(
            AccountTransferAmount transferAmount, Set<Long> writableCampaignIds) {

        ItemValidationBuilder<AccountTransferAmount, Defect> ivb = ItemValidationBuilder.of(transferAmount);

        ivb.item(transferAmount.getAccountId(), "accountId")
                .check(notNull())
                .check(validId())
                .check(inSet(writableCampaignIds));

        ivb.item(transferAmount.getAmount(), "amount")
                .check(notNull())
                .check(greaterThan(BigDecimal.ZERO));

        return ivb.getResult();
    }

}
