package ru.yandex.direct.intapi.entity.gorynych;

import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import javax.annotation.Nonnull;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.advq.query.IllegalQueryException;
import ru.yandex.direct.core.entity.stopword.service.StopWordService;
import ru.yandex.direct.intapi.entity.gorynych.model.CheckRequest;
import ru.yandex.direct.intapi.entity.gorynych.model.CheckRequestGroup;
import ru.yandex.direct.intapi.entity.gorynych.model.CheckResponse;
import ru.yandex.direct.intapi.entity.gorynych.model.CheckResponseGroup;
import ru.yandex.direct.intapi.entity.gorynych.model.CheckResponseInclusion;
import ru.yandex.direct.intapi.validation.IntApiDefect;
import ru.yandex.direct.intapi.webapp.semaphore.Semaphore;
import ru.yandex.direct.libs.keywordutils.KeywordUtils;
import ru.yandex.direct.libs.keywordutils.StopWordMatcher;
import ru.yandex.direct.libs.keywordutils.helper.ParseKeywordCache;
import ru.yandex.direct.libs.keywordutils.inclusion.KeywordInclusionUtils;
import ru.yandex.direct.libs.keywordutils.inclusion.model.KeywordForInclusion;
import ru.yandex.direct.libs.keywordutils.inclusion.model.KeywordWithLemmasFactory;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.intapi.validation.IntApiConstraints.listSize;
import static ru.yandex.direct.intapi.validation.IntApiConstraints.notNull;
import static ru.yandex.direct.intapi.validation.ValidationUtils.checkResult;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@RestController
@Api(value = "Проверки включений минус-фраз во ключевые фразы")
@RequestMapping(value = "/gorynych/keywords-inclusion")
@Semaphore(permits = 10, key = "gorynych/keywords-inclusion", retryTimeout = 3_000)
public class KeywordsInclusionController {
    private static final Logger logger = LoggerFactory.getLogger(KeywordsInclusionController.class);
    private static final int MAX_GROUPS_COUNT = 1000;
    private static final int MAX_MINUS_PHRASES_COUNT = 10_000;
    private static final int MAX_PLUS_PHRASES_COUNT = 10_000;

    private final StopWordMatcher stopWordsMatcher;
    private final KeywordWithLemmasFactory keywordFactory;
    private final ParseKeywordCache parseKeywordCache;

    public KeywordsInclusionController(StopWordService stopWordService,
                                       KeywordWithLemmasFactory keywordFactory,
                                       ParseKeywordCache parseKeywordCache) {
        this.stopWordsMatcher = stopWordService::isStopWord;
        this.keywordFactory = keywordFactory;
        this.parseKeywordCache = parseKeywordCache;
    }

    @ApiOperation(
            value = "Для каждой ключвой фразы получить список минус-фраз, которые вычитают"
                    + " ключевик целиком == будут проигнорированы",
            notes = "Результаты верхнего уровня сохраняют порядок запроса. "
                    + "Порядок ключевиков / минус-фраз негарантирван, но их текст сохраняется полностью.",
            httpMethod = "POST"
    )
    @RequestMapping(path = "check",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_UTF8_VALUE
    )
    public CheckResponse check(
            @ApiParam(value = "запрос", required = true)
            @RequestBody @Nonnull CheckRequest request
    ) {
        ValidationResult<CheckRequest, IntApiDefect> vr = validateCheckRequest(request);
        checkResult(vr);

        return new CheckResponse(mapList(request.getGroups(), this::processGroup));
    }

    // обработка одной группы запроса
    CheckResponseGroup processGroup(CheckRequestGroup group) {
        // оригиналы фраз
        Map<KeywordForInclusion, String> origs = new IdentityHashMap<>();

        List<KeywordForInclusion> plusKeywords = makeKeywords(group.getPlusKeywords(), origs);
        List<KeywordForInclusion> minusKeywords = makeKeywords(group.getMinusKeywords(), origs);

        List<CheckResponseInclusion> inclusions =
                EntryStream.of(KeywordInclusionUtils
                        .getPlusKeywordsGivenEmptyResult(stopWordsMatcher, plusKeywords, minusKeywords))
                        .mapKeyValue(
                                (plusKw, minusKws) -> new CheckResponseInclusion(
                                        origs.get(plusKw),
                                        mapList(minusKws, origs::get)
                                )
                        )
                        .toList();
        return new CheckResponseGroup(inclusions);
    }


    // валидация запроса
    ValidationResult<CheckRequest, IntApiDefect> validateCheckRequest(CheckRequest request) {
        ItemValidationBuilder<CheckRequest, IntApiDefect> v = ItemValidationBuilder.of(request, IntApiDefect.class);

        if (request == null) {
            v.check(notNull());
            return v.getResult();
        }

        v.list(request.getGroups(), "groups")
                .check(notNull())
                .checkEach(notNull())
                .check(listSize(0, MAX_GROUPS_COUNT))
                .checkEachBy(this::validateCheckRequestGroup);

        return v.getResult();
    }

    // валидация одной группы из запроса
    private ValidationResult<CheckRequestGroup, IntApiDefect> validateCheckRequestGroup(CheckRequestGroup group) {
        ItemValidationBuilder<CheckRequestGroup, IntApiDefect> v = ItemValidationBuilder.of(group, IntApiDefect.class);
        if (group != null) {
            v.list(group.getMinusKeywords(), "minus_phrases")
                    .check(notNull())
                    .check(listSize(0, MAX_MINUS_PHRASES_COUNT))
                    .checkEach(notNull())
                    .checkEach(validKeyword());
            v.list(group.getPlusKeywords(), "plus_phrases")
                    .check(notNull())
                    .check(listSize(0, MAX_PLUS_PHRASES_COUNT))
                    .checkEach(notNull())
                    .checkEach(validKeyword());
        }
        return v.getResult();
    }

    // неэффективная проверка попыткой распарсить, но это самый простой вариант
    private Constraint<String, IntApiDefect> validKeyword() {
        return val -> {
            try {
                parseKeywordCache.parse(val);
            } catch (IllegalQueryException ex) {
                return defectInfo -> String.format("incorrect phrase %s: %s", defectInfo.getPath(), ex.getMessage());
            } catch (RuntimeException ex) {
                logger.error("incorrect phrase %s: unknown error", ex);
                return defectInfo -> String.format("incorrect phrase %s: unknown error", defectInfo.getPath());
            }
            return null;
        };
    }

    // Парсим строки в KeywordForInclusion, попутно запоминаем оригиналы в origs
    private List<KeywordForInclusion> makeKeywords(List<String> keywords, Map<KeywordForInclusion, String> origs) {
        return StreamEx.of(keywords)
                .mapToEntry(parseKeywordCache::parse, Function.identity())
                .mapKeys(kw -> KeywordUtils.stripStopWords(kw, stopWordsMatcher))
                .mapKeys(keywordFactory::keywordFrom)
                .peekKeyValue(origs::put)
                .keys()
                .toList();
    }
}
