package ru.yandex.direct.intapi.entity.idm;

import java.util.Map;

import javax.ws.rs.core.MediaType;

import com.fasterxml.jackson.core.type.TypeReference;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.entity.idm.model.AddRoleRequest;
import ru.yandex.direct.intapi.entity.idm.model.GetAllRolesResponse;
import ru.yandex.direct.intapi.entity.idm.model.IdmResponse;
import ru.yandex.direct.intapi.entity.idm.model.InfoResponse;
import ru.yandex.direct.intapi.entity.idm.model.RemoveRoleRequest;
import ru.yandex.direct.intapi.entity.idm.model.RoleAdditionalFields;
import ru.yandex.direct.intapi.entity.idm.service.IdmUserManagementIntapiService;
import ru.yandex.direct.tvm.AllowServices;

import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;
import static ru.yandex.direct.intapi.entity.idm.converter.IdmCommonNames.SLUG;
import static ru.yandex.direct.tvm.TvmService.DIRECT_AUTOTESTS;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.IDM_PROD;
import static ru.yandex.direct.tvm.TvmService.IDM_TEST;
import static ru.yandex.direct.utils.JsonUtils.fromJson;

@RestController
@RequestMapping("idm")
@AllowServices(production = {IDM_PROD}, testing = {IDM_TEST, DIRECT_DEVELOPER, DIRECT_AUTOTESTS})
public class IdmRolesController {

    private static final TypeReference<Map<String, String>> ROLE_TYPE_REFERENCE = new TypeReference<>() {
    };

    private final IdmUserManagementIntapiService idmUserManagementIntapiService;

    @Autowired
    public IdmRolesController(IdmUserManagementIntapiService idmUserManagementIntapiService) {
        this.idmUserManagementIntapiService = idmUserManagementIntapiService;
    }

    @ApiOperation(
            value = "получить список всех ролей",
            nickname = "get-all-roles",
            httpMethod = "GET"
    )
    @ApiResponses({
            @ApiResponse(code = 403, message = "Недостаточно прав для выполнения операции",
                    response = ErrorResponse.class),
            @ApiResponse(code = 400, message = "Некорректный запрос",
                    response = ErrorResponse.class)
    })
    @RequestMapping(path = "get-all-roles",
            method = RequestMethod.GET,
            produces = MediaType.APPLICATION_JSON
    )
    public GetAllRolesResponse getAllRoles() {
        return idmUserManagementIntapiService.getAllRoles();
    }

    @ApiOperation(
            value = "получить роли по доменному логину",
            nickname = "get-user-roles",
            httpMethod = "GET"
    )
    @ApiResponses({
            @ApiResponse(code = 403, message = "Недостаточно прав для выполнения операции",
                    response = ErrorResponse.class),
            @ApiResponse(code = 400, message = "Некорректный запрос",
                    response = ErrorResponse.class)
    })
    @RequestMapping(path = "get-user-roles",
            method = RequestMethod.GET,
            produces = MediaType.APPLICATION_JSON
    )
    public IdmResponse getUserRoles(
            @RequestParam(value = "login", required = false) String domainLogin) {
        return idmUserManagementIntapiService.getUserBasicRoles(domainLogin);
    }

    @ApiOperation(
            value = "получить информацию о доступных ролях",
            nickname = "info",
            httpMethod = "GET"
    )
    @ApiResponses({
            @ApiResponse(code = 403, message = "Недостаточно прав для выполнения операции",
                    response = ErrorResponse.class),
            @ApiResponse(code = 400, message = "Некорректный запрос",
                    response = ErrorResponse.class)
    })
    @RequestMapping(path = "info",
            method = RequestMethod.GET,
            produces = MediaType.APPLICATION_JSON
    )
    public InfoResponse info() {
        return idmUserManagementIntapiService.info();
    }

    @ApiOperation(
            value = "добавление роли",
            nickname = "add-role",
            httpMethod = "POST"
    )
    @ApiResponses({
            @ApiResponse(code = 403, message = "Недостаточно прав для выполнения операции",
                    response = ErrorResponse.class),
            @ApiResponse(code = 400, message = "Некорректный запрос",
                    response = ErrorResponse.class)
    })
    @RequestMapping(path = "add-role",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_FORM_URLENCODED,
            produces = MediaType.APPLICATION_JSON
    )
    public IdmResponse addRole(
            @RequestParam("role") String jsonRole,
            @RequestParam("fields") String jsonAdditionalFields,
            @RequestParam(value = "login", required = false) String domainLogin,
            @RequestParam(value = "group", required = false) Long groupId) {

        String roleName = fromJson(jsonRole, ROLE_TYPE_REFERENCE).get(SLUG);
        RoleAdditionalFields additionalFields = fromJson(jsonAdditionalFields, RoleAdditionalFields.class);

        AddRoleRequest request = new AddRoleRequest()
                .withRole(roleName)
                .withDomainLogin(domainLogin)
                .withGroupId(groupId)
                .withPassportLogin(additionalFields.getPassportLogin())
                .withClientId(additionalFields.getClientId());

        return idmUserManagementIntapiService.addRole(request);
    }

    @ApiOperation(
            value = "удаление роли",
            nickname = "remove-role",
            httpMethod = "POST"
    )
    @ApiResponses({
            @ApiResponse(code = 403, message = "Недостаточно прав для выполнения операции",
                    response = ErrorResponse.class),
            @ApiResponse(code = 400, message = "Некорректный запрос",
                    response = ErrorResponse.class)
    })
    @RequestMapping(path = "remove-role",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_FORM_URLENCODED,
            produces = MediaType.APPLICATION_JSON
    )
    public IdmResponse removeRole(
            @RequestParam("role") String jsonRole,
            @RequestParam(value = "data", required = false) String data,
            @RequestParam(value = "fields", required = false) String fields,
            @RequestParam(value = "login", required = false) String domainLogin,
            @RequestParam(value = "group", required = false) Long groupId) {

        String roleName = fromJson(jsonRole, ROLE_TYPE_REFERENCE).get(SLUG);

        String jsonAdditionalFields = defaultIfNull(data, fields);
        RoleAdditionalFields additionalFields = fromJson(jsonAdditionalFields, RoleAdditionalFields.class);

        RemoveRoleRequest request = new RemoveRoleRequest()
                .withRole(roleName)
                .withDomainLogin(domainLogin)
                .withGroupId(groupId)
                .withPassportLogin(additionalFields.getPassportLogin())
                .withClientId(additionalFields.getClientId());

        return idmUserManagementIntapiService.removeRole(request);
    }
}
