package ru.yandex.direct.intapi.entity.idm.converter;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.client.model.ClientPrimaryManager;
import ru.yandex.direct.intapi.entity.idm.model.GetAllRolesResponseGroupsItem;
import ru.yandex.direct.intapi.entity.idm.model.GetAllRolesResponseItem;
import ru.yandex.direct.intapi.entity.idm.model.GetUserRolesResponse;
import ru.yandex.direct.intapi.entity.idm.model.IdmClientGroup;
import ru.yandex.direct.intapi.entity.idm.model.IdmSupportForClientRole;
import ru.yandex.direct.intapi.entity.idm.model.IdmUserRole;

import static java.util.Arrays.asList;
import static java.util.Collections.singletonMap;
import static ru.yandex.direct.intapi.entity.idm.converter.IdmCommonNames.CLIENT_ID_SLUG;
import static ru.yandex.direct.intapi.entity.idm.converter.IdmCommonNames.MAIN_MANAGER_ROLE_SLUG_NAME;
import static ru.yandex.direct.intapi.entity.idm.converter.IdmCommonNames.PASSPORT_LOGIN;
import static ru.yandex.direct.intapi.entity.idm.converter.IdmCommonNames.SLUG;
import static ru.yandex.direct.intapi.entity.idm.converter.IdmCommonNames.SUPPORT_FOR_CLIENT_ROLE_NAME;

@ParametersAreNonnullByDefault
public class GetAllRolesResponseConverter {
    private GetAllRolesResponseConverter() {
    }

    public static List<GetAllRolesResponseItem> convertUserRoles(
            List<IdmUserRole> basicRoles,
            List<ClientPrimaryManager> mainManagerRoles,
            List<IdmSupportForClientRole> supportForClientRoles) {

        List<GetAllRolesResponseItem> basicItems = convertBasicRoles(basicRoles);
        List<GetAllRolesResponseItem> mainManagerItems = convertMainManagerRoles(mainManagerRoles);
        List<GetAllRolesResponseItem> supportForClientItems = convertSupportForClientRoles(supportForClientRoles);

        List<GetAllRolesResponseItem> allItems = StreamEx.of(basicItems, mainManagerItems, supportForClientItems)
                .flatMap(Collection::stream)
                .toList();

        Map<String, List<List<Map<String, String>>>> rolesByLogin = StreamEx.of(allItems)
                .mapToEntry(GetAllRolesResponseItem::getLogin, GetAllRolesResponseItem::getRoles)
                .flatMapValues(Collection::stream)
                .grouping();

        return EntryStream.of(rolesByLogin)
                .mapKeyValue((login, roles) -> new GetAllRolesResponseItem()
                        .withLogin(login)
                        .withRoles(roles))
                .toList();
    }

    private static List<GetAllRolesResponseItem> convertBasicRoles(List<IdmUserRole> userRoles) {
        Map<String, List<List<Map<String, String>>>> rolesByLogin = StreamEx.of(userRoles)
                .mapToEntry(IdmUserRole::getDomainLogin,
                        ur -> asList(
                                singletonMap(SLUG, ur.getRole().getTypedValue()),
                                singletonMap(PASSPORT_LOGIN, ur.getPassportLogin())))
                .grouping();

        return EntryStream.of(rolesByLogin)
                .mapKeyValue((login, roles) -> new GetAllRolesResponseItem()
                        .withLogin(login)
                        .withRoles(roles))
                .toList();
    }

    public static GetUserRolesResponse convertBasicRolesForOneLogin(List<IdmUserRole> userRoles) {
        List<List<Map<String, String>>> roles = StreamEx.of(userRoles)
                .map(ur -> asList(
                        singletonMap(SLUG, ur.getRole().getTypedValue()),
                        singletonMap(PASSPORT_LOGIN, ur.getPassportLogin())))
                .toList();

        return new GetUserRolesResponse()
                .withRoles(roles);
    }

    public static List<GetAllRolesResponseItem> convertMainManagerRoles(List<ClientPrimaryManager> userRoles) {
        Map<String, List<List<Map<String, String>>>> rolesByLogin = StreamEx.of(userRoles)
                .mapToEntry(ClientPrimaryManager::getDomainLogin,
                        ur -> asList(singletonMap(SLUG, MAIN_MANAGER_ROLE_SLUG_NAME),
                                Map.of(PASSPORT_LOGIN, ur.getPassportLogin(),
                                        CLIENT_ID_SLUG, ur.getSubjectClientId().toString())))
                .grouping();

        return EntryStream.of(rolesByLogin)
                .mapKeyValue((login, roles) -> new GetAllRolesResponseItem()
                        .withLogin(login)
                        .withRoles(roles))
                .toList();
    }

    private static List<GetAllRolesResponseItem> convertSupportForClientRoles(List<IdmSupportForClientRole> userRoles) {
        Map<String, List<List<Map<String, String>>>> rolesByLogin = StreamEx.of(userRoles)
                .mapToEntry(IdmSupportForClientRole::getDomainLogin,
                        ur -> asList(
                                singletonMap(SLUG, SUPPORT_FOR_CLIENT_ROLE_NAME),
                                Map.of(
                                        PASSPORT_LOGIN, ur.getPassportLogin(),
                                        CLIENT_ID_SLUG, ur.getSubjectClientId().toString())))
                .grouping();

        return EntryStream.of(rolesByLogin)
                .mapKeyValue((login, roles) -> new GetAllRolesResponseItem()
                        .withLogin(login)
                        .withRoles(roles))
                .toList();
    }

    public static List<GetAllRolesResponseGroupsItem> convertGroupRoles(List<IdmClientGroup> groupRoles) {
        Map<Long, List<List<Map<String, String>>>> rolesByGroupId = StreamEx.of(groupRoles)
                .mapToEntry(IdmClientGroup::getIdmGroupId,
                        gr -> asList(
                                singletonMap(SLUG, gr.getRole().getTypedValue()),
                                singletonMap(CLIENT_ID_SLUG, Long.toString(gr.getClientId().asLong()))))
                .grouping();

        return EntryStream.of(rolesByGroupId)
                .mapKeyValue((groupId, roles) -> new GetAllRolesResponseGroupsItem()
                        .withGroup(groupId)
                        .withRoles(roles))
                .toList();
    }
}
