package ru.yandex.direct.intapi.entity.idm.memberships.service;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.http.NameValuePair;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.message.BasicNameValuePair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.idm.model.IdmGroup;
import ru.yandex.direct.core.entity.idm.model.IdmGroupMember;
import ru.yandex.direct.core.entity.idm.model.IdmGroupMemberId;
import ru.yandex.direct.core.entity.idm.model.IdmRequiredRole;
import ru.yandex.direct.core.entity.idm.repository.IdmGroupsRepository;
import ru.yandex.direct.core.entity.idm.service.IdmGroupsMembersService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.intapi.entity.idm.memberships.container.MemberCheckResult;
import ru.yandex.direct.intapi.entity.idm.memberships.container.MembershipsPageInfo;
import ru.yandex.direct.intapi.entity.idm.memberships.converter.MembershipConverter;
import ru.yandex.direct.intapi.entity.idm.memberships.model.ChangeMembershipsResponse;
import ru.yandex.direct.intapi.entity.idm.memberships.model.GetMembershipsResponse;
import ru.yandex.direct.intapi.entity.idm.memberships.model.Membership;
import ru.yandex.direct.intapi.entity.idm.memberships.model.MembershipError;
import ru.yandex.direct.rbac.RbacRole;

import static java.util.function.Function.identity;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.intapi.entity.idm.memberships.controller.IdmMembershipsController.GET_MEMBERSHIPS_BASE_PATH;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class IdmMembershipsService {
    /**
     * Код, что операция выполнена полностью без ошибок.
     */
    private static final int OK = 0;
    /**
     * Код, что операция выполнена частично, есть ошибки.
     */
    private static final int PARTIAL_OK = 207;
    private static final Integer DEFAULT_LIMIT = 1000;
    private static final String EMPTY_NEXT_URL = null;

    private final UserService userService;
    private final IdmGroupsRepository idmGroupsRepository;
    private final IdmGroupsMembersService idmGroupsMembersService;

    @Autowired
    public IdmMembershipsService(UserService userService,
                                 IdmGroupsRepository idmGroupsRepository,
                                 IdmGroupsMembersService idmGroupsMembersService) {
        this.userService = userService;
        this.idmGroupsRepository = idmGroupsRepository;
        this.idmGroupsMembersService = idmGroupsMembersService;
    }


    public ChangeMembershipsResponse addMemberships(List<Membership> memberships) {
        Collection<User> users = getUsers(memberships);
        List<IdmGroupMember> idmGroupMembers = convertToGroupMembers(memberships, users);
        List<MemberCheckResult> memberCheckResults = StreamEx.of(idmGroupMembers)
                .map(MemberCheckResult::new)
                .toList();
        validateMembershipsInfo(memberCheckResults);
        validateMembershipsGroupRoles(memberCheckResults);
        validateMembershipsUserRoles(memberCheckResults, users);

        addValidMembers(memberCheckResults);

        return convertToResponse(memberCheckResults);
    }

    private Collection<User> getUsers(List<Membership> memberships) {
        List<String> logins = mapList(memberships, Membership::getPassportLogin);
        return userService.massGetUserByLogin(logins);
    }

    private void addValidMembers(List<MemberCheckResult> memberCheckResults) {
        List<IdmGroupMember> validMembers = StreamEx.of(memberCheckResults)
                .filter(m -> !m.hasAnyErrors())
                .map(MemberCheckResult::getMember)
                .toList();
        idmGroupsMembersService.addMembersWhichNotExist(validMembers);
    }

    public ChangeMembershipsResponse removeMemberships(List<Membership> memberships) {
        Collection<User> users = getUsers(memberships);
        List<IdmGroupMember> idmGroupMembers = convertToGroupMembers(memberships, users);
        List<MemberCheckResult> memberCheckResults = StreamEx.of(idmGroupMembers)
                .map(MemberCheckResult::new)
                .toList();
        validateMembershipsInfo(memberCheckResults);
        removeValidMembers(memberCheckResults);
        return convertToResponse(memberCheckResults);
    }

    /**
     * Постранично возвращает всех членов всех групп.
     */
    public GetMembershipsResponse getMemberships(MembershipsPageInfo pageInfo) {
        int pageLimit = Optional.ofNullable(pageInfo.getRecordsLimit())
                .filter(v -> v > 0)
                .orElse(DEFAULT_LIMIT);
        List<IdmGroupMember> nextMembersPage = getPagedMemberships(pageInfo, pageLimit);
        List<Membership> memberships = mapList(nextMembersPage, MembershipConverter::toIdm);
        String nextUrl = getNextUrl(pageLimit, nextMembersPage);
        return new GetMembershipsResponse()
                .withCode(0)
                .withMemberships(memberships)
                .withNextUrl(nextUrl);
    }

    private List<IdmGroupMember> getPagedMemberships(MembershipsPageInfo pageInfo, int pageLimit) {
        IdmGroupMemberId lastMemberId = pageInfo.getGroupId() != null && pageInfo.getClientId() != null ?
                new IdmGroupMember()
                        .withIdmGroupId(pageInfo.getGroupId())
                        .withClientId(ClientId.fromLong(pageInfo.getClientId()))
                : null;
        return idmGroupsMembersService.getNextMembersPage(lastMemberId, pageLimit);
    }

    private String getNextUrl(int pageLimit, List<IdmGroupMember> nextMembersPage) {
        if (nextMembersPage.size() < pageLimit) {
            return EMPTY_NEXT_URL;
        }
        IdmGroupMember last = Iterables.getLast(nextMembersPage);
        List<NameValuePair> valuePairs = EntryStream
                .of(MembershipsPageInfo.CLIENT_ID, last.getClientId(),
                        MembershipsPageInfo.GROUP_ID, last.getIdmGroupId(),
                        MembershipsPageInfo.LIMIT, pageLimit)
                .nonNullValues()
                .mapValues(Object::toString)
                .mapKeyValue(BasicNameValuePair::new)
                .map(NameValuePair.class::cast)
                .toList();
        URIBuilder ub = new URIBuilder()
                .setPath(GET_MEMBERSHIPS_BASE_PATH)
                .addParameters(valuePairs);
        return ub.toString();
    }

    private void removeValidMembers(List<MemberCheckResult> memberCheckResults) {
        List<IdmGroupMember> validMembers = StreamEx.of(memberCheckResults)
                .filter(r -> !r.hasAnyErrors())
                .map(MemberCheckResult::getMember)
                .toList();
        idmGroupsMembersService.removeMembers(validMembers);
    }

    private List<IdmGroupMember> convertToGroupMembers(List<Membership> memberships, Collection<User> users) {
        Map<String, User> usersByLogins = listToMap(users, User::getLogin);
        return StreamEx.of(memberships)
                .map(m -> {
                    String passportLogin = m.getPassportLogin();
                    User user = usersByLogins.get(passportLogin);
                    return new IdmGroupMember()
                            .withIdmGroupId(m.getGroupId().longValue())
                            .withLogin(passportLogin)
                            .withDomainLogin(m.getDomainLogin())
                            .withClientId(user != null ? user.getClientId() : null)
                            .withUid(user != null ? user.getUid() : null);
                })
                .toList();
    }

    private ChangeMembershipsResponse convertToResponse(List<MemberCheckResult> memberCheckResults) {
        List<MembershipError> membershipErrors = StreamEx.of(memberCheckResults)
                .filter(MemberCheckResult::hasAnyErrors)
                .map(result -> {
                    IdmGroupMember member = result.getMember();
                    String errors = String.join("; ", result.getErrors());
                    return new MembershipError()
                            .withGroup(member.getIdmGroupId().intValue())
                            .withDomainLogin(member.getDomainLogin())
                            .withPassportLogin(member.getLogin())
                            .withErrorMessage(errors);
                })
                .toList();
        boolean isNoErrors = isEmpty(membershipErrors);
        return new ChangeMembershipsResponse()
                .withCode(isNoErrors ? OK : PARTIAL_OK)
                .withErrors(isNoErrors ? null : membershipErrors);
    }

    private void validateMembershipsInfo(List<MemberCheckResult> memberCheckResults) {
        memberCheckResults.forEach(result -> {
            IdmGroupMember member = result.getMember();
            List<String> errors = result.getErrors();
            if (member.getUid() == null) {
                errors.add(String.format("User with login='%s' is not found", member.getLogin()));
            } else if (member.getClientId() == null) {
                errors.add(String.format("Client for user with login='%s' is not found", member.getLogin()));
            }
        });
    }

    private void validateMembershipsGroupRoles(List<MemberCheckResult> memberCheckResults) {
        Set<Long> groupIds = listToSet(memberCheckResults, m -> m.getMember().getIdmGroupId());
        List<IdmGroup> groupRoles = idmGroupsRepository.getGroups(groupIds);
        Map<Long, IdmRequiredRole> roleByGroupIds =
                listToMap(groupRoles, IdmGroup::getIdmGroupId, IdmGroup::getRequiredRole);
        memberCheckResults.forEach(result -> {
            IdmGroupMember member = result.getMember();
            List<String> errors = result.getErrors();
            Long idmGroupId = member.getIdmGroupId();
            if (idmGroupId == null) {
                errors.add("GroupId is not set");
            } else if (roleByGroupIds.get(idmGroupId) == null) {
                errors.add(String.format("Group with GroupId=%d is not found", idmGroupId));
            } else if (!Objects.equals(roleByGroupIds.get(idmGroupId), IdmRequiredRole.MANAGER)) {
                errors.add(String.format("Group with GroupId=%d is not a manager group", idmGroupId));
            }
        });
    }

    private void validateMembershipsUserRoles(List<MemberCheckResult> memberCheckResults, Collection<User> users) {
        List<MemberCheckResult> filledMemberResults =
                filterList(memberCheckResults,
                        r -> r.getMember().getClientId() != null && r.getMember().getUid() != null);
        Map<Long, User> usersByUids = listToMap(users, User::getUid, identity());
        for (MemberCheckResult result : filledMemberResults) {
            IdmGroupMember member = result.getMember();
            Long uid = member.getUid();
            User user = usersByUids.get(uid);
            String login = member.getLogin();
            List<String> errors = result.getErrors();
            if (user == null) {
                errors.add(String.format("User with login='%s' is not found", login));
                continue;
            }
            if (user.getDomainLogin() == null || !user.getRole().isInternal()) {
                errors.add(String.format("User with uid=%d is not internal user", uid));
            } else {
                String domainLogin = member.getDomainLogin();
                if (!Objects.equals(domainLogin, user.getDomainLogin())) {
                    errors.add(String.format("Domain login '%s' is not valid", domainLogin));
                }
            }
            if (!Objects.equals(user.getRole(), RbacRole.MANAGER)) {
                errors.add(String.format("User with login='%s' is not manager", login));
            }
        }
    }

}
