package ru.yandex.direct.intapi.entity.idm.service;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.blackbox.client.BlackboxClient;
import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.core.entity.client.service.AddClientOptions;
import ru.yandex.direct.core.entity.client.service.AddClientService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.repository.UserRepository;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.core.entity.user.service.validation.UserValidationService;
import ru.yandex.direct.core.service.integration.balance.BalanceService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.LoginOrUid;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.intapi.entity.idm.model.AddRoleRequest;
import ru.yandex.direct.intapi.entity.idm.model.IdmFatalResponse;
import ru.yandex.direct.intapi.entity.idm.model.IdmResponse;
import ru.yandex.direct.intapi.entity.idm.model.IdmRole;
import ru.yandex.direct.intapi.entity.idm.model.RoleWithTeamleaders;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.rbac.PpcRbac;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacSubrole;
import ru.yandex.direct.regions.Region;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.staff.client.StaffClient;
import ru.yandex.direct.staff.client.model.json.Name;
import ru.yandex.direct.staff.client.model.json.PersonInfo;
import ru.yandex.direct.staff.client.model.json.RuEnValue;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxCorrectResponse;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static org.apache.commons.collections4.CollectionUtils.isNotEmpty;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isAnyTeamLeader;
import static ru.yandex.direct.intapi.entity.idm.model.AddRoleResponse.addRoleResponseWithPassportLogin;
import static ru.yandex.direct.intapi.entity.idm.model.IdmException.idmErrorException;
import static ru.yandex.direct.intapi.entity.idm.model.IdmException.idmFatalException;
import static ru.yandex.direct.intapi.entity.idm.service.IdmGetRolesService.RBAC_ROLES_BY_IDM_ROLES;
import static ru.yandex.direct.intapi.entity.idm.service.IdmRolesUtils.getRoleWithTeamleaders;
import static ru.yandex.direct.intapi.entity.idm.service.IdmRolesUtils.isRoleAnyTeamLeader;
import static ru.yandex.direct.intapi.entity.idm.service.IdmRolesUtils.isRoleDeveloper;
import static ru.yandex.direct.intapi.entity.idm.service.IdmRolesUtils.isRoleSuperReader;
import static ru.yandex.direct.intapi.entity.idm.service.IdmRolesUtils.processIdmRequestSafe;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.JsonUtils.toJson;
import static ru.yandex.direct.utils.PassportUtils.normalizeLogin;

@Service
@ParametersAreNonnullByDefault
public class IdmAddRoleService {

    private static final Logger logger = LoggerFactory.getLogger(IdmAddRoleService.class);

    // YandexOffice::get_default_office()->{office_id}
    private static final Long DEFAULT_MANAGER_OFFICE_ID = 1L;

    private final UserService userService;
    private final PpcRbac ppcRbac;
    private final AddClientService addClientService;
    private final ClientService clientService;
    private final UserRepository userRepository;
    private final ShardHelper shardHelper;
    private final StaffClient staffClient;
    private final UserValidationService userValidationService;
    private final BlackboxClient blackboxClient;
    private final BalanceService balanceService;
    private final TvmIntegration tvmIntegration;
    private final TvmService tvmService;

    @Autowired
    @SuppressWarnings("checkstyle:parameternumber")
    public IdmAddRoleService(UserService userService,
                             PpcRbac ppcRbac,
                             AddClientService addClientService,
                             ClientService clientService,
                             UserRepository userRepository,
                             ShardHelper shardHelper,
                             StaffClient staffClient,
                             UserValidationService userValidationService,
                             BlackboxClient blackboxClient,
                             BalanceService balanceService,
                             TvmIntegration tvmIntegration,
                             EnvironmentType environmentType) {
        this.userService = userService;
        this.ppcRbac = ppcRbac;
        this.addClientService = addClientService;
        this.clientService = clientService;
        this.userRepository = userRepository;
        this.shardHelper = shardHelper;
        this.staffClient = staffClient;
        this.userValidationService = userValidationService;
        this.blackboxClient = blackboxClient;
        this.balanceService = balanceService;
        this.tvmIntegration = tvmIntegration;
        this.tvmService = environmentType.isProductionOrPrestable()
                ? TvmService.BLACKBOX_PROD
                : TvmService.BLACKBOX_MIMINO;
    }

    public IdmResponse addRole(AddRoleRequest request) {
        return processIdmRequestSafe(() -> addRoleInternal(request), logger);
    }

    private IdmResponse addRoleInternal(AddRoleRequest request) {
        logger.info("request: {}", toJson(request));

        String validationErrorText =
                preValidate(request.getRole(), request.getDomainLogin(), request.getPassportLogin());
        if (validationErrorText != null) {
            logger.info("addRole validation error: {}", validationErrorText);
            return new IdmFatalResponse(validationErrorText);
        }

        String newIdmRoleName = Objects.requireNonNull(request.getRole());
        IdmRole newIdmRole = IdmRole.fromTypedValue(newIdmRoleName.toLowerCase());

        String passportLogin = normalizeLogin(request.getPassportLogin());
        String domainLogin = normalizeLogin(request.getDomainLogin());

        User staffInfo = getStaffInfoAndCheck(domainLogin);
        Long userId = getUserIdAndCheck(passportLogin);

        User user = userService.getUser(userId);
        // чтобы была понятная ошибка вначале проверяем роль, потом что у клиента нет других представителей
        checkRoleCanBeChanged(passportLogin, newIdmRole, user);
        checkClientDoesNotHaveOtherRepresentatives(userId);

        if ((user == null) || (user.getRole() == RbacRole.EMPTY)) {
            return addRoleForNewClient(passportLogin, newIdmRole, domainLogin, user, staffInfo);
        } else {
            return addRoleForExistingClient(passportLogin, newIdmRole, domainLogin, user);
        }
    }

    private IdmResponse addRoleForNewClient(String passportLogin, IdmRole newIdmRole, String domainLogin,
                                            @Nullable User user, User staffInfo) {
        boolean newRoleIsAnyTeamLeader = isRoleAnyTeamLeader(newIdmRole);

        if (newRoleIsAnyTeamLeader) {
            throw idmFatalException(String.format("логин %s не имеет роли менеджера (текущая роль: %s)",
                    passportLogin, RoleWithTeamleaders.EMPTY.name().toLowerCase()));
        }

        RbacRole role = RBAC_ROLES_BY_IDM_ROLES.get(newIdmRole);
        UidAndClientId uidAndClientId;

        if (user == null) {
            // создает пользователя с указанной ролью в БД и Балансе
            uidAndClientId = createClientWithRole(passportLogin, role, staffInfo.getFio());
        } else {
            clientService.updateClientRole(user.getClientId(), role, null);
            uidAndClientId = UidAndClientId.of(user.getUid(), user.getClientId());
        }

        boolean shouldSetDeveloperFlag = isRoleDeveloper(newIdmRole) || isRoleSuperReader(newIdmRole);
        if (shouldSetDeveloperFlag) {
            boolean isDeveloper = isRoleDeveloper(newIdmRole);
            userService.setDeveloperFlag(uidAndClientId.getUid(), isDeveloper);
        }

        saveInternalUsersParams(uidAndClientId, domainLogin, role);
        updateFioEmailPhone(uidAndClientId, staffInfo);

        return addRoleResponseWithPassportLogin(passportLogin);
    }

    private UidAndClientId createClientWithRole(String passportLogin, RbacRole role, String fio) {
        AddClientOptions addClientOptions = AddClientOptions.defaultOptions()
                .withUseExistingIfRequired(true);

        Result<UidAndClientId> addClientResult = addClientService.processRequest(
                LoginOrUid.of(passportLogin), fio, Region.RUSSIA_REGION_ID, CurrencyCode.RUB, role,
                addClientOptions);

        if (!addClientResult.isSuccessful()) {
            logger.warn("add client errors: {}", addClientResult.getErrors());
            throw idmErrorException("внутренняя ошибка сервера");
        }
        return addClientResult.getResult();
    }

    private IdmResponse addRoleForExistingClient(String passportLogin, IdmRole newIdmRole, String domainLogin,
                                                 User user) {
        boolean newRoleIsAnyTeamLeader = isRoleAnyTeamLeader(newIdmRole);

        RoleWithTeamleaders newRole = RoleWithTeamleaders.fromIdmRole(newIdmRole);
        RoleWithTeamleaders oldRole = getRoleWithTeamleaders(user);

        // роль уже есть
        if (oldRole == newRole || (newRole == RoleWithTeamleaders.MANAGER && isAnyTeamLeader(user))) {
            // если запрашиваемый логин был заблокирован (удален через управлятор), то разблокируем
            if (user.getStatusBlocked()) {
                unblockUser(user, domainLogin);
            }
            return addRoleResponseWithPassportLogin(passportLogin);
        }

        // для тимлидера уже должен существовать менеджер и он не должен состоять ни в одной группе тимлидеров
        if (newRoleIsAnyTeamLeader && (oldRole == RoleWithTeamleaders.MANAGER)) {

            Optional<Long> managerSupervisor = ppcRbac.getManagerSupervisor(user.getUid());
            if (managerSupervisor.isPresent()) {
                throw idmFatalException(String.format("Логин %s является подчиненным другого тимлидера. " +
                                "Для получения тимлидерского доступа нужно исключить логин из всех групп менеджеров",
                        passportLogin));
            }
            RbacSubrole subrole = RoleWithTeamleaders.TEAMLEADER.equals(newRole)
                    ? RbacSubrole.TEAMLEADER
                    : RbacSubrole.SUPERTEAMLEADER;
            clientService.updateClientRole(user.getClientId(), RbacRole.MANAGER, subrole);

            return addRoleResponseWithPassportLogin(passportLogin);
        }
        throw idmFatalException(String.format("логин %s уже имеет роль: %s", passportLogin,
                oldRole.name().toLowerCase()));
    }

    private void saveInternalUsersParams(UidAndClientId uidAndClientId, String domainLogin, RbacRole role) {
        Long userId = uidAndClientId.getUid();
        ClientId clientId = uidAndClientId.getClientId();
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        userRepository.updateDomainLogin(shard, userId, domainLogin);

        if (role == RbacRole.MANAGER) {
            userRepository.addManagerInHierarchy(shard, clientId, userId);
            userRepository.setManagerOfficeId(shard, userId, DEFAULT_MANAGER_OFFICE_ID);
        }
    }

    private void unblockUser(User user, String domainLogin) {
        // ситуация разблокировки со сменой доменного логина спорная и её поддержку отложили на будущее
        if (!domainLogin.equalsIgnoreCase(user.getDomainLogin())) {
            throw idmFatalException(String.format("переданный доменный логин %s отличается от %s",
                    domainLogin, user.getDomainLogin()));
        }
        userService.unblockUser(user.getClientId(), user.getUid());
    }

    private void updateFioEmailPhone(UidAndClientId uidAndClientId, User staffInfo) {
        User targetUser = checkNotNull(userService.getUser(uidAndClientId.getUid()));

        // валидацию staffInfo делаем до вызова этого метода
        ModelChanges<User> changes = convertToChanges(targetUser.getUid(), staffInfo);
        AppliedChanges<User> userAppliedChanges = changes.applyTo(targetUser);

        userService.update(userAppliedChanges);
    }

    private ModelChanges<User> convertToChanges(Long uid, User staffInfo) {
        ModelChanges<User> changes = new ModelChanges<>(uid, User.class);

        changes.processNotNull(staffInfo.getFio(), User.FIO);
        changes.processNotNull(staffInfo.getEmail(), User.EMAIL);
        changes.process(staffInfo.getPhone(), User.PHONE);

        return changes;
    }

    private User getStaffInfoAndCheck(String domainLogin) {
        User staffInfo = getStaffInfo(domainLogin);
        if (staffInfo == null) {
            throw idmFatalException(String.format("доменный логин '%s' не найден на staff", domainLogin));
        }
        ValidationResult<User, Defect> userVr = userValidationService.validateUser(staffInfo);
        if (userVr.hasAnyErrors()) {
            logger.warn("validate user errors: {}", userVr.flattenErrors());
            throw idmErrorException("ФИО или email не заданы на staff");
        }
        return staffInfo;
    }

    @Nullable
    private User getStaffInfo(String domainLogin) {
        Map<String, PersonInfo> personInfoByLogin = staffClient.getStaffUserInfos(singletonList(domainLogin));
        PersonInfo personInfo = personInfoByLogin.get(domainLogin);
        return convertStaffInfo(personInfo);
    }

    @Nullable
    private static User convertStaffInfo(@Nullable PersonInfo personInfo) {
        if (personInfo == null) {
            return null;
        }

        Optional<Name> name = Optional.ofNullable(personInfo.getName());
        String lastName = name.map(Name::getLast).map(RuEnValue::getRu).orElse("");
        String firstName = name.map(Name::getFirst).map(RuEnValue::getRu).orElse("");
        String fio = lastName + " " + firstName;

        return new User()
                .withFio(fio)
                .withEmail(personInfo.getWorkEmail())
                .withPhone(ifNotNull(personInfo.getWorkPhone(), phone -> Integer.toString(phone)));
    }

    private Long getUserIdAndCheck(String passportLogin) {
        Optional<Long> userId = getUidByLoginFromBlackbox(passportLogin);
        return userId.orElseThrow(
                () -> idmErrorException(String.format("паспортный логин '%s' не найден", passportLogin)));
    }

    private void checkClientDoesNotHaveOtherRepresentatives(Long userId) {
        ClientId clientId = balanceService.findClientIdByUid(userId).orElse(null);
        if (clientId != null) {

            List<Long> allUserIdsOfClient = userService.massGetUidsByClientIds(singletonList(clientId))
                    .getOrDefault(clientId, emptyList());
            List<Long> otherUserIds = filterList(allUserIdsOfClient, id -> !Objects.equals(id, userId));

            if (isNotEmpty(otherUserIds)) {
                throw idmFatalException("add-role: uid associated with ClientID in Balance " +
                        "and this ClientID already associated with another uid");
            }
        }
    }

    private Optional<Long> getUidByLoginFromBlackbox(String passportLogin) {
        String tvmTicket = tvmIntegration.getTicket(tvmService);
        BlackboxCorrectResponse user = blackboxClient.userInfo(HttpUtil.getRemoteAddressForBlackbox(),
                passportLogin, Cf.list(), tvmTicket);
        return user.getUid().toOptional().map(PassportUid::getUid);
    }

    private void checkRoleCanBeChanged(String passportLogin, IdmRole newIdmRole, @Nullable User user) {
        boolean newRoleIsAnyTeamLeader = isRoleAnyTeamLeader(newIdmRole);

        RoleWithTeamleaders newRole = RoleWithTeamleaders.fromIdmRole(newIdmRole);
        RoleWithTeamleaders oldRole = Optional.ofNullable(user).map(IdmRolesUtils::getRoleWithTeamleaders)
                .orElse(RoleWithTeamleaders.EMPTY);

        if ((oldRole != RoleWithTeamleaders.EMPTY)
                && (oldRole != newRole)
                && !(newRole == RoleWithTeamleaders.MANAGER && isAnyTeamLeader(user))
                && !(newRoleIsAnyTeamLeader && (oldRole == RoleWithTeamleaders.MANAGER))) {
            throw idmFatalException(String.format("логин %s уже имеет роль: %s", passportLogin,
                    oldRole.name().toLowerCase()));
        }
    }

    @Nullable
    static String preValidate(@Nullable String roleName,
                              @Nullable String domainLogin,
                              @Nullable String passportLogin) {
        if (roleName == null) {
            return "не задана роль";
        } else if (!IdmRole.isValidRole(roleName.toLowerCase())) {
            return String.format("ошибка в названии роли: '%s'", roleName);
        }
        if (isBlank(domainLogin)) {
            return "требуется доменный логин";
        }
        if (isBlank(passportLogin)) {
            return "паспортный логин не задан";
        }
        return null;
    }
}
