package ru.yandex.direct.intapi.entity.idm.service;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import one.util.streamex.EntryStream;
import org.apache.http.NameValuePair;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.message.BasicNameValuePair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.intapi.entity.idm.container.LastReturnedRoleInfo;
import ru.yandex.direct.intapi.entity.idm.container.PageInfo;
import ru.yandex.direct.intapi.entity.idm.controller.IdmRoleManagementController;
import ru.yandex.direct.intapi.entity.idm.model.GetRolesResponse;
import ru.yandex.direct.intapi.entity.idm.model.GetRolesResponseItem;
import ru.yandex.direct.intapi.entity.idm.service.typesupport.GroupRolesPagedSupplier;
import ru.yandex.direct.intapi.entity.idm.service.typesupport.MainManagerRolesPagedSupplier;
import ru.yandex.direct.intapi.entity.idm.service.typesupport.PersonalRolesPagedSupplier;
import ru.yandex.direct.intapi.entity.idm.service.typesupport.RolesPagedSupplier;
import ru.yandex.direct.intapi.entity.idm.service.typesupport.SupportForClientRolesPagedSupplier;

@Service
@ParametersAreNonnullByDefault
public class IdmGetPagedRolesService {
    private static final Integer DEFAULT_LIMIT = 1000;
    private static final String EMPTY_NEXT_URL = null;

    private final List<RolesPagedSupplier> rolesPagedSuppliers;

    @Autowired
    public IdmGetPagedRolesService(PersonalRolesPagedSupplier personalRolesPagedSupplier,
                                   SupportForClientRolesPagedSupplier supportForClientRolesPagedSupplier,
                                   GroupRolesPagedSupplier groupRolesPagedSupplier,
                                   MainManagerRolesPagedSupplier mainManagerRolesPagedSupplier) {
        rolesPagedSuppliers = List.of(personalRolesPagedSupplier, supportForClientRolesPagedSupplier,
                groupRolesPagedSupplier, mainManagerRolesPagedSupplier);
    }

    public GetRolesResponse getRoles(PageInfo pageInfo) {
        int pageLimit = Optional.ofNullable(pageInfo.getRecordsLimit())
                .filter(v -> v > 0)
                .orElse(DEFAULT_LIMIT);
        List<GetRolesResponseItem> items = getRoleItems(pageInfo, pageLimit);
        String nextUrl = getNextUrl(pageLimit, items);
        return new GetRolesResponse()
                .withNextUrl(nextUrl)
                .withRoles(items);
    }

    private LastReturnedRoleInfo getLastReturnedRoleInfo(PageInfo pageInfo) {
        return new LastReturnedRoleInfo()
                .withRoleName(pageInfo.getRoleName())
                .withClientId(pageInfo.getClientId())
                .withManagerUid(pageInfo.getManagerUid())
                .withSupportId(pageInfo.getSupportId())
                .withGroupId(pageInfo.getGroupId());
    }

    private List<GetRolesResponseItem> getRoleItems(PageInfo pageInfo, int pageSize) {
        ArrayList<GetRolesResponseItem> items = new ArrayList<>(pageSize);
        LastReturnedRoleInfo lastRole = getLastReturnedRoleInfo(pageInfo);
        LastReturnedRoleInfo emptyLastRole = new LastReturnedRoleInfo();
        Iterator<RolesPagedSupplier> supplierIterator = rolesPagedSuppliers.iterator();
        int currentLimit = pageSize - items.size();
        while (supplierIterator.hasNext() && 0 < currentLimit) {
            RolesPagedSupplier rolesSupplier = supplierIterator.next();
            if (lastRole.getRoleName() != null && !rolesSupplier.isSupportedRole(lastRole.getRoleName())) {
                continue;
            }
            List<GetRolesResponseItem> roles = rolesSupplier.getNextPage(lastRole, currentLimit);
            items.addAll(roles);
            currentLimit = pageSize - items.size();
            lastRole = emptyLastRole;
        }
        return items;
    }

    private String getNextUrl(int pageLimit, List<GetRolesResponseItem> items) {
        if (items.size() < pageLimit) {
            return EMPTY_NEXT_URL;
        }
        GetRolesResponseItem last = Iterables.getLast(items);
        List<NameValuePair> valuePairs = EntryStream
                .of(PageInfo.ROLE_NAME, last.getRoleName(),
                        PageInfo.CLIENT_ID, last.getClientId(),
                        PageInfo.MANAGER_UID, last.getInternalUid(),
                        PageInfo.SUPPORT_ID, last.getSupportId(),
                        PageInfo.GROUP_ID, last.getGroupId(),
                        PageInfo.LIMIT, pageLimit)
                .nonNullValues()
                .mapValues(Object::toString)
                .mapKeyValue(BasicNameValuePair::new)
                .map(NameValuePair.class::cast)
                .toList();
        URIBuilder ub = new URIBuilder()
                .setPath(IdmRoleManagementController.GET_ROLES_BASE_PATH)
                .addParameters(valuePairs);
        return ub.toString();
    }

}
