package ru.yandex.direct.intapi.entity.idm.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.repository.UserRepository;
import ru.yandex.direct.dbschema.ppc.enums.ClientsRole;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.intapi.entity.idm.model.IdmRole;
import ru.yandex.direct.intapi.entity.idm.model.IdmUserRole;
import ru.yandex.direct.rbac.RbacRole;

import static ru.yandex.direct.core.entity.user.utils.UserUtil.isDeveloper;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isSuperTeamLeader;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isTeamLeader;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@Service
@ParametersAreNonnullByDefault
public class IdmGetRolesService {

    private static final Logger logger = LoggerFactory.getLogger(IdmGetRolesService.class);

    private static final ImmutableMap<RbacRole, IdmRole> IDM_ROLES_BY_RBAC_ROLES =
            new ImmutableMap.Builder<RbacRole, IdmRole>()
                    .put(RbacRole.SUPER, IdmRole.SUPER)
                    .put(RbacRole.SUPERREADER, IdmRole.SUPERREADER)
                    .put(RbacRole.SUPPORT, IdmRole.SUPPORT)
                    .put(RbacRole.PLACER, IdmRole.PLACER)
                    .put(RbacRole.MEDIA, IdmRole.MEDIA)
                    .put(RbacRole.MANAGER, IdmRole.MANAGER)
                    .put(RbacRole.INTERNAL_AD_ADMIN, IdmRole.INTERNAL_AD_ADMIN)
                    .put(RbacRole.INTERNAL_AD_MANAGER, IdmRole.INTERNAL_AD_MANAGER)
                    .put(RbacRole.INTERNAL_AD_SUPERREADER, IdmRole.INTERNAL_AD_SUPERREADER)
                    .put(RbacRole.LIMITED_SUPPORT, IdmRole.LIMITED_SUPPORT)
                    .build();

    static final ImmutableMap<IdmRole, RbacRole> RBAC_ROLES_BY_IDM_ROLES =
            ImmutableMap.copyOf(EntryStream.of(IDM_ROLES_BY_RBAC_ROLES).invert()
                    .append(IdmRole.DEVELOPER, RbacRole.SUPERREADER).toMap());

    private static final String DEFAULT_LOGIN = "-";

    private final ShardHelper shardHelper;
    private final ClientRepository clientRepository;
    private final UserRepository userRepository;

    @Autowired
    public IdmGetRolesService(
            ShardHelper shardHelper,
            ClientRepository clientRepository,
            UserRepository userRepository) {
        this.shardHelper = shardHelper;
        this.clientRepository = clientRepository;
        this.userRepository = userRepository;
    }

    /**
     * Получение списка всех ролей пользователей
     */
    public List<IdmUserRole> getAllRoles() {
        return StreamEx.of(shardHelper.dbShards())
                .map(this::getAllRolesSharded)
                .toFlatList(Function.identity());
    }

    private List<IdmUserRole> getAllRolesSharded(int shard) {
        Set<ClientsRole> clientsRoles = mapSet(IDM_ROLES_BY_RBAC_ROLES.keySet(), RbacRole::toSource);
        List<Long> chiefUids = clientRepository.getClientChiefUidsWithRoles(shard, clientsRoles);
        return getRolesByUserIds(shard, chiefUids);
    }

    public List<IdmUserRole> getUserRoles(String domainLogin) {
        return StreamEx.of(shardHelper.dbShards())
                .map(shard -> getUserRolesSharded(shard, domainLogin))
                .toFlatList(Function.identity());
    }

    private List<IdmUserRole> getUserRolesSharded(int shard, String domainLogin) {
        Set<Long> userIds = userRepository.getUidsByDomainLogin(shard, domainLogin);
        return getRolesByUserIds(shard, userIds);
    }

    private List<IdmUserRole> getRolesByUserIds(int shard, Collection<Long> userIds) {
        List<User> users = userRepository.fetchByUids(shard, userIds);
        List<User> filteredUsers = StreamEx.of(users)
                .filter(user -> !user.getStatusBlocked())
                .filter(IdmGetRolesService::isValidRole)
                .toList();

        List<IdmUserRole> userRoles = new ArrayList<>();
        for (User user : filteredUsers) {

            String domainLogin = defaultLoginIfEmpty(user.getDomainLogin());
            String passportLogin = defaultLoginIfEmpty(user.getLogin());

            IdmRole role = IDM_ROLES_BY_RBAC_ROLES.get(user.getRole());

            // для тимлидеров возвращаем и их базовую роль менеджера и тимлидерскую (DIRECT-17723)
            if (isTeamLeader(user)) {
                userRoles.add(idmUserRole(domainLogin, passportLogin, user.getUid(), role));
                userRoles.add(idmUserRole(domainLogin, passportLogin, user.getUid(), IdmRole.TEAMLEADER));
            } else if (isSuperTeamLeader(user)) {
                userRoles.add(idmUserRole(domainLogin, passportLogin, user.getUid(), role));
                userRoles.add(idmUserRole(domainLogin, passportLogin, user.getUid(), IdmRole.SUPERTEAMLEADER));
            } else if (isDeveloper(user)) {
                userRoles.add(idmUserRole(domainLogin, passportLogin, user.getUid(), IdmRole.DEVELOPER));
            } else {
                userRoles.add(idmUserRole(domainLogin, passportLogin, user.getUid(), role));
            }
        }
        Set<IdmRole> enabledRoleNames = getEnabledRoleNames();
        return StreamEx.of(userRoles)
                .filter(ur -> enabledRoleNames.contains(ur.getRole()))
                .toList();
    }

    private static boolean isValidRole(User user) {
        RbacRole role = user.getRole();
        if ((role != null) && IDM_ROLES_BY_RBAC_ROLES.containsKey(role)) {
            return true;
        }
        logger.warn("invalid role for uid={}, role={}", user.getUid(), role);
        return false;
    }

    static String defaultLoginIfEmpty(@Nullable String login) {
        return StringUtils.defaultIfEmpty(login, DEFAULT_LOGIN).toLowerCase();
    }

    private static IdmUserRole idmUserRole(String domainLogin, String passportLogin, Long uid, IdmRole role) {
        return new IdmUserRole()
                .withDomainLogin(domainLogin)
                .withPassportLogin(passportLogin)
                .withUid(uid)
                .withRole(role);
    }

    Set<IdmRole> getEnabledRoleNames() {
        return Set.of(
                IdmRole.SUPER, IdmRole.SUPERREADER, IdmRole.SUPPORT, IdmRole.PLACER, IdmRole.MEDIA,
                IdmRole.MANAGER, IdmRole.TEAMLEADER, IdmRole.SUPERTEAMLEADER,
                IdmRole.INTERNAL_AD_ADMIN, IdmRole.INTERNAL_AD_MANAGER,
                IdmRole.INTERNAL_AD_SUPERREADER,
                IdmRole.DEVELOPER,
                IdmRole.LIMITED_SUPPORT
        );
    }
}
