package ru.yandex.direct.intapi.entity.idm.service;

import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.idm.model.IdmGroup;
import ru.yandex.direct.core.entity.idm.model.IdmGroupRole;
import ru.yandex.direct.core.entity.idm.model.IdmRequiredRole;
import ru.yandex.direct.core.entity.idm.repository.IdmGroupsRepository;
import ru.yandex.direct.core.entity.idm.repository.IdmGroupsRolesRepository;
import ru.yandex.direct.core.entity.idm.service.IdmGroupsRolesService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.intapi.entity.idm.model.AddRoleRequest;
import ru.yandex.direct.intapi.entity.idm.model.IdmClientGroup;
import ru.yandex.direct.intapi.entity.idm.model.IdmFatalResponse;
import ru.yandex.direct.intapi.entity.idm.model.IdmGroupRoleName;
import ru.yandex.direct.intapi.entity.idm.model.IdmResponse;
import ru.yandex.direct.intapi.entity.idm.model.IdmSuccessResponse;
import ru.yandex.direct.intapi.entity.idm.model.RemoveRoleRequest;
import ru.yandex.direct.rbac.RbacRole;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.intapi.entity.idm.model.AddRoleResponse.addRoleResponseWithClientId;
import static ru.yandex.direct.intapi.entity.idm.service.IdmRolesUtils.processIdmRequestSafe;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.isValidId;
import static ru.yandex.direct.utils.JsonUtils.toJson;

@Service
@ParametersAreNonnullByDefault
public class IdmGroupRolesService {

    private static final Logger logger = LoggerFactory.getLogger(IdmGroupRolesService.class);
    private static final Set<RbacRole> CLIENT_REQUIRED_ROLES = Set.of(RbacRole.AGENCY, RbacRole.CLIENT);

    private final IdmGroupsRolesRepository idmGroupsRolesRepository;
    private final IdmGroupsRepository idmGroupsRepository;
    private final IdmGroupsRolesService idmGroupsRolesService;
    private final ShardHelper shardHelper;
    private final ClientService clientService;


    @Autowired
    public IdmGroupRolesService(
            IdmGroupsRolesRepository idmGroupsRolesRepository,
            IdmGroupsRepository idmGroupsRepository,
            IdmGroupsRolesService idmGroupsRolesService,
            ShardHelper shardHelper,
            ClientService clientService) {
        this.idmGroupsRolesRepository = idmGroupsRolesRepository;
        this.idmGroupsRepository = idmGroupsRepository;
        this.idmGroupsRolesService = idmGroupsRolesService;
        this.shardHelper = shardHelper;
        this.clientService = clientService;
    }

    /**
     * Получение списка всех групповых ролей
     */
    public List<IdmClientGroup> getAllGroupRoles() {
        List<IdmGroupRole> allRoles = idmGroupsRolesService.getAllRoles();

        return StreamEx.of(allRoles)
                .map(r -> new IdmClientGroup()
                        .withIdmGroupId(r.getIdmGroupId())
                        .withClientId(r.getClientId())
                        .withRole(IdmGroupRoleName.MANAGER_FOR_CLIENT))
                .toList();
    }

    public IdmResponse addGroupRole(AddRoleRequest request) {
        return processIdmRequestSafe(() -> addGroupRoleInternal(request), logger);
    }

    private IdmResponse addGroupRoleInternal(AddRoleRequest request) {
        logger.info("request: {}", toJson(request));

        Long groupId = request.getGroupId();
        ClientId clientId = ClientId.fromNullableLong(request.getClientId());

        String validationErrorText = validate(groupId, clientId);
        if (validationErrorText != null) {
            logger.info("addRole validation error: {}", validationErrorText);
            return new IdmFatalResponse(validationErrorText);
        }

        addIdmGroupIfAbsent(groupId);
        addIdmGroupClientRelation(clientId, groupId);

        return addRoleResponseWithClientId(clientId.asLong());
    }

    public IdmResponse removeGroupRole(RemoveRoleRequest request) {
        return processIdmRequestSafe(() -> removeGroupRoleInternal(request), logger);
    }

    private IdmResponse removeGroupRoleInternal(RemoveRoleRequest request) {
        logger.info("request: {}", toJson(request));

        Long groupId = request.getGroupId();
        ClientId clientId = ClientId.fromNullableLong(request.getClientId());

        String validationErrorText = validate(groupId, clientId);
        if (validationErrorText != null) {
            logger.info("removeRole validation error: {}", validationErrorText);
            return new IdmFatalResponse(validationErrorText);
        }
        removeIdmGroupClientRelation(clientId, groupId);

        return new IdmSuccessResponse();
    }

    private void addIdmGroupIfAbsent(Long idmGroupId) {
        List<IdmGroup> existingGroups = idmGroupsRepository.getGroups(singletonList(idmGroupId));

        if (existingGroups.isEmpty()) {
            IdmGroup groupToAdd = new IdmGroup()
                    .withIdmGroupId(idmGroupId)
                    .withRequiredRole(IdmRequiredRole.MANAGER);
            idmGroupsRepository.add(singletonList(groupToAdd));
        }
    }

    private void addIdmGroupClientRelation(ClientId subjectClientId, Long idmGroupId) {
        int shard = shardHelper.getShardByClientIdStrictly(subjectClientId);
        IdmGroupRole idmGroupRole = new IdmGroupRole()
                .withIdmGroupId(idmGroupId)
                .withClientId(subjectClientId);
        idmGroupsRolesRepository.addRolesWhichNotExist(shard, singletonList(idmGroupRole));
    }

    private void removeIdmGroupClientRelation(ClientId subjectClientId, Long idmGroupId) {
        int shard = shardHelper.getShardByClientIdStrictly(subjectClientId);
        idmGroupsRolesRepository.removeRole(shard, subjectClientId, idmGroupId);
    }

    private String validate(@Nullable Long groupId, @Nullable ClientId clientId) {
        if (!isValidId(groupId)) {
            return "group required";
        }
        Client client = ifNotNull(clientId, clientService::getClient);

        if (client == null) {
            return String.format("can't find client by given clientId %s", clientId);
        }
        if (client.getRole() == null || !CLIENT_REQUIRED_ROLES.contains(client.getRole())) {
            return String.format("client with client_id %s has wrong role", clientId);
        }
        return null;
    }
}
