package ru.yandex.direct.intapi.entity.idm.service;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.model.ClientPrimaryManager;
import ru.yandex.direct.core.entity.client.service.ClientPrimaryManagerService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.intapi.entity.idm.model.AddRoleRequest;
import ru.yandex.direct.intapi.entity.idm.model.IdmErrorResponse;
import ru.yandex.direct.intapi.entity.idm.model.IdmResponse;
import ru.yandex.direct.intapi.entity.idm.model.IdmSuccessResponse;
import ru.yandex.direct.intapi.entity.idm.model.RemoveRoleRequest;
import ru.yandex.direct.intapi.validation.kernel.TranslatableIntapiDefect;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.presentation.DefectPresentationRegistry;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectInfo;

import static org.apache.commons.lang3.StringUtils.isBlank;
import static ru.yandex.direct.intapi.common.converter.DefectsConverter.defectsToString;
import static ru.yandex.direct.intapi.entity.idm.model.AddRoleResponse.addRoleResponse;
import static ru.yandex.direct.utils.JsonUtils.toJson;

@Service
@ParametersAreNonnullByDefault
public class IdmMainManagerService {

    private static final Logger logger = LoggerFactory.getLogger(IdmMainManagerService.class);

    private final ClientPrimaryManagerService clientPrimaryManagerService;
    private final ClientService clientService;
    private final UserService userService;
    private final TranslationService translationService;
    private final DefectPresentationRegistry<TranslatableIntapiDefect> defectPresentationRegistry;

    @Autowired
    public IdmMainManagerService(ClientPrimaryManagerService clientPrimaryManagerService, ClientService clientService,
                                 UserService userService,
                                 TranslationService translationService,
                                 DefectPresentationRegistry<TranslatableIntapiDefect> defectPresentationRegistry) {
        this.clientPrimaryManagerService = clientPrimaryManagerService;
        this.clientService = clientService;
        this.userService = userService;
        this.translationService = translationService;
        this.defectPresentationRegistry = defectPresentationRegistry;
    }

    public IdmResponse addRole(AddRoleRequest request) {
        logger.info("addRole request: {}", toJson(request));
        MainManagerInfoContainer info = new MainManagerInfoContainer(request.getPassportLogin(), request.getClientId())
                .initFields();
        List<String> errors = validate(info);
        if (!errors.isEmpty()) {
            return new IdmErrorResponse(String.join("; ", errors));
        }
        ClientPrimaryManager newManager = new ClientPrimaryManager()
                .withSubjectClientId(ClientId.fromLong(request.getClientId()))
                .withPrimaryManagerUid(info.getManager().getUid())
                .withPassportLogin(request.getPassportLogin())
                .withDomainLogin(request.getDomainLogin())
                .withIsIdmPrimaryManager(true);
        Result<Long> result = clientPrimaryManagerService.updatePrimaryManager(newManager);
        if (result.isSuccessful()) {
            return addRoleResponse(request.getPassportLogin(), request.getClientId());
        }
        return getErrorResponse(result);
    }

    public IdmResponse removeRole(RemoveRoleRequest request) {
        logger.info("removeRole request: {}", toJson(request));
        MainManagerInfoContainer info = new MainManagerInfoContainer(request.getPassportLogin(), request.getClientId())
                .initFields();
        List<String> errors = validate(info);
        if (!errors.isEmpty()) {
            return new IdmErrorResponse(String.join("; ", errors));
        }
        // Если текущий менеджер не тот, что в запросе, то ничего не делаем, всё уже сделано, сразу возвращаем OK.
        // В противном случае сбрасываем текущего менеджера в NULL.
        User manager = info.getManager();
        Client subjectClient = info.getSubjectClient();
        if (!Objects.equals(subjectClient.getPrimaryManagerUid(), manager.getUid())) {
            return new IdmSuccessResponse();
        }
        ClientPrimaryManager newManager = new ClientPrimaryManager()
                .withSubjectClientId(ClientId.fromLong(request.getClientId()))
                .withPrimaryManagerUid(null)
                .withIsIdmPrimaryManager(false);
        Result<Long> result = clientPrimaryManagerService.updatePrimaryManager(newManager);
        if (result.isSuccessful()) {
            return new IdmSuccessResponse();
        }
        return getErrorResponse(result);
    }

    private IdmResponse getErrorResponse(Result<Long> result) {
        List<DefectInfo<Defect>> defectInfos = result.getValidationResult().flattenErrors();
        String errors = defectsToString(translationService, defectPresentationRegistry, defectInfos);
        return new IdmErrorResponse(errors);
    }

    private List<String> validate(MainManagerInfoContainer info) {
        List<String> errors = new ArrayList<>();
        if (isBlank(info.getPassportLogin())) {
            errors.add("Passport login cannot be empty.");
        } else if (info.getManager() == null) {
            errors.add(String.format("User with passport login '%s' not found.", info.getPassportLogin()));
        }
        if (info.getSubjectClient() == null) {
            errors.add(String.format("Client with clientId=%d not found.", info.getSubjectClientId()));
        }
        return errors;
    }

    public List<ClientPrimaryManager> getAllRoles() {
        return clientPrimaryManagerService.getAllIdmPrimaryManagers();
    }

    private class MainManagerInfoContainer {
        private final String passportLogin;
        private final Long subjectClientId;
        private User manager;
        private Client subjectClient;

        MainManagerInfoContainer(@Nullable String passportLogin, @Nullable Long subjectClientId) {
            this.passportLogin = passportLogin;
            this.subjectClientId = subjectClientId;
        }

        @Nullable
        public String getPassportLogin() {
            return passportLogin;
        }

        @Nullable
        public Long getSubjectClientId() {
            return subjectClientId;
        }

        public User getManager() {
            return manager;
        }

        public Client getSubjectClient() {
            return subjectClient;
        }

        MainManagerInfoContainer initFields() {
            manager = Optional.ofNullable(passportLogin)
                    .map(userService::getUserByLogin)
                    .orElse(null);
            subjectClient = Optional.ofNullable(subjectClientId)
                    .map(ClientId::fromLong)
                    .map(clientService::getClient)
                    .orElse(null);
            return this;
        }
    }


}
