package ru.yandex.direct.intapi.entity.idm.service;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.logmail.service.LogMailService;
import ru.yandex.direct.intapi.entity.idm.model.RemoveRoleEmailParameters;
import ru.yandex.direct.mail.EmailAddress;
import ru.yandex.direct.mail.MailMessage;
import ru.yandex.direct.mail.MailSender;

@ParametersAreNonnullByDefault
@Service
public class IdmNotificationMailSenderService {

    private static final String EMAIL_FROM = "direct_noreply@yandex-team.ru";
    private static final String EMAIL_TO = "account-exchange@yandex-team.ru";
    private static final EmailAddress EMAIL_ADDRESS_FROM = new EmailAddress(EMAIL_FROM, "Yandex.Direct");
    private static final EmailAddress EMAIL_ADDRESS_TO = new EmailAddress(EMAIL_TO, null);

    private static final String REMOVE_ROLE_SUBJECT = "Яндекс.Директ/Увольнение сотрудника";

    private final MailSender mailSender;
    private final LogMailService logMailService;

    @Autowired
    public IdmNotificationMailSenderService(MailSender mailSender,
                                            LogMailService logMailService) {
        this.mailSender = mailSender;
        this.logMailService = logMailService;
    }

    /**
     * Отправляет уведомление о необходимости перенести всех клиентов с менеджера при увольнении.
     */
    public void sendRemoveRoleEmail(RemoveRoleEmailParameters parameters) {
        MailMessage emailContent = createRemoveRoleEmail(parameters);

        // Сохраняем адрес получателя письма, чтобы использовать при логгировании письма. Делается для того,
        // чтобы на девтесте в логи вставлялся правильный адрес, а не direct-dev-letters@.
        String emailTo = emailContent.getTo().getEmail();

        mailSender.send(emailContent);
        logMailService.addIdmRemoveRoleLogMail(emailTo, emailContent.getSubject(), emailContent.getMessageBody());
    }

    private MailMessage createRemoveRoleEmail(RemoveRoleEmailParameters parameters) {
        String messageBody = "Здравствуйте!\n"
                + "\n"
                + "При удалении роли \"" + parameters.getRoleName() + "\" произошла ошибка:\n"
                + "  " + parameters.getErrorText() + "\n"
                + "\n"
                + "Данные по сотруднику:\n"
                + "Логин: " + parameters.getManagerLogin() + "\n"
                + "Доменный логин: " + parameters.getDomainLogin() + "\n"
                + "Клиенты менеджера: https://direct.yandex.ru/registered/main.pl?cmd=showManagerMyClients&ulogin="
                + parameters.getManagerLogin() + "\n"
                + "\n"
                + "--\n"
                + "С уважением,\n"
                + "Отдел продаж компании \"Яндекс\"\n"
                + "support@direct.yandex.ru" + "\n"
                + "http://www.yandex.ru";

        return new MailMessage(EMAIL_ADDRESS_FROM, EMAIL_ADDRESS_TO, REMOVE_ROLE_SUBJECT, messageBody);
    }
}
