package ru.yandex.direct.intapi.entity.idm.service;

import java.util.List;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.model.ManagerHierarchyInfo;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.client.service.ManagerHierarchyService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.intapi.entity.idm.model.IdmFatalResponse;
import ru.yandex.direct.intapi.entity.idm.model.IdmResponse;
import ru.yandex.direct.intapi.entity.idm.model.IdmRole;
import ru.yandex.direct.intapi.entity.idm.model.IdmSuccessResponse;
import ru.yandex.direct.intapi.entity.idm.model.IdmWarningResponse;
import ru.yandex.direct.intapi.entity.idm.model.RemoveRoleEmailParameters;
import ru.yandex.direct.intapi.entity.idm.model.RemoveRoleRequest;
import ru.yandex.direct.intapi.entity.idm.model.RoleWithTeamleaders;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.rbac.PpcRbac;
import ru.yandex.direct.rbac.RbacClientsRelations;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.model.ClientsRelation;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.intapi.entity.idm.service.IdmAddRoleService.preValidate;
import static ru.yandex.direct.intapi.entity.idm.service.IdmRolesUtils.isRoleAnySuperReader;
import static ru.yandex.direct.intapi.entity.idm.service.IdmRolesUtils.isRoleAnyTeamLeader;
import static ru.yandex.direct.intapi.entity.idm.service.IdmRolesUtils.processIdmRequestSafe;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.JsonUtils.toJson;

@Service
@ParametersAreNonnullByDefault
public class IdmRemoveRoleService {

    private static final Logger logger = LoggerFactory.getLogger(IdmRemoveRoleService.class);

    private final UserService userService;
    private final ClientService clientService;
    private final PpcRbac ppcRbac;
    private final ManagerHierarchyService managerHierarchyService;
    private final IdmNotificationMailSenderService mailSenderService;
    private final RbacClientsRelations rbacClientsRelations;

    @Autowired
    public IdmRemoveRoleService(UserService userService,
                                ClientService clientService,
                                PpcRbac ppcRbac,
                                ManagerHierarchyService managerHierarchyService,
                                IdmNotificationMailSenderService mailSenderService,
                                RbacClientsRelations rbacClientsRelations) {
        this.userService = userService;
        this.clientService = clientService;
        this.ppcRbac = ppcRbac;
        this.managerHierarchyService = managerHierarchyService;
        this.mailSenderService = mailSenderService;
        this.rbacClientsRelations = rbacClientsRelations;
    }

    public IdmResponse removeRole(RemoveRoleRequest request) {
        return processIdmRequestSafe(() -> removeRoleInternal(request), logger);
    }

    private IdmResponse removeRoleInternal(RemoveRoleRequest request) {
        logger.info("request: {}", toJson(request));

        String validationErrorText =
                preValidate(request.getRole(), request.getDomainLogin(), request.getPassportLogin());
        if (validationErrorText != null) {
            logger.info("removeRole validation error: {}", validationErrorText);
            return new IdmFatalResponse(validationErrorText);
        }

        String domainLogin = Objects.requireNonNull(request.getDomainLogin());
        String passportLogin = Objects.requireNonNull(request.getPassportLogin());

        String idmRoleToRemoveName = Objects.requireNonNull(request.getRole());
        IdmRole idmRoleToRemove = IdmRole.fromTypedValue(idmRoleToRemoveName.toLowerCase());
        RoleWithTeamleaders roleToRemove = RoleWithTeamleaders.fromIdmRole(idmRoleToRemove);

        User user = userService.getUserByLogin(passportLogin);

        RoleWithTeamleaders oldRole = ifNotNull(user, IdmRolesUtils::getRoleWithTeamleaders);
        String oldDomainLogin = ifNotNull(user, User::getDomainLogin);

        // логин уже не имеет роли
        if ((oldRole == null) || (oldRole == RoleWithTeamleaders.EMPTY)) {
            logger.info("remove_role: 0, login: {}, role: {}, login_is_empty: 1", passportLogin, roleToRemove);
            return new IdmSuccessResponse();
        }

        // DIRECT-22783: отвечать Управлятору «ок» при попытке удаления менеджерской роли у тимлидера
        if ((roleToRemove == RoleWithTeamleaders.MANAGER) &&
                (oldRole == RoleWithTeamleaders.TEAMLEADER || oldRole == RoleWithTeamleaders.SUPERTEAMLEADER)) {
            logger.info("remove_role: 0, login: {}, role: {}, remove_manager_for_teamleaders_login: 1",
                    passportLogin, roleToRemove);
            return new IdmSuccessResponse();
        }

        // DIRECT-15289: отвечать Управлятору «ок» на попытку отзыва роли, которой нет в Директе
        if (roleToRemove != oldRole) {
            logger.info("remove_role: 0, login: {}, role: {}, role_not_equal_rbac_role: 1",
                    passportLogin, roleToRemove);
            return new IdmSuccessResponse();
        }

        // DIRECT-12497: если доменный логин в директе уже сменили, то управлятору сообщаем что роль успешно удалили
        if (!domainLogin.equalsIgnoreCase(oldDomainLogin)) {
            logger.info("remove_role: 0, login: {}, role: {}, domain_logins_different: 1", passportLogin, roleToRemove);
            return new IdmSuccessResponse();
        }


        if (isRoleAnyTeamLeader(idmRoleToRemove)) {
            return removeAnyTeamLeaderRole(user, idmRoleToRemove);

        } else if (isRoleAnySuperReader(idmRoleToRemove)) {
            return removeAnySuperReaderRole(user, idmRoleToRemove);

        } else if (idmRoleToRemove == IdmRole.MANAGER) {
            return removeManagerRole(user, idmRoleToRemove);

        } else if (idmRoleToRemove == IdmRole.LIMITED_SUPPORT) {
            return removeLimitedSupportRole(user, idmRoleToRemove);

        } else {
            return defaultRemoveRole(user, idmRoleToRemove);
        }
    }

    private IdmResponse defaultRemoveRole(User user, IdmRole role) {
        clientService.updateClientRole(user.getClientId(), RbacRole.EMPTY, null);
        logger.info("remove_role: 1, login: {}, role: {}", user.getLogin(), role);
        return new IdmSuccessResponse();
    }

    private IdmResponse removeAnySuperReaderRole(User user, IdmRole role) {
        userService.setDeveloperFlag(user.getUid(), false);
        return defaultRemoveRole(user, role);
    }

    private IdmResponse removeAnyTeamLeaderRole(User user, IdmRole role) {
        List<Long> subordinateUids = ppcRbac.getSupervisorDirectSubordinates(user.getUid());

        if (subordinateUids.isEmpty()) {
            clientService.updateClientRole(user.getClientId(), RbacRole.MANAGER, null);
            logger.info("remove_role: 1, login: {}, role: {}", user.getLogin(), role);
            return new IdmSuccessResponse();

        } else {
            String errorText = IdmRole.TEAMLEADER.equals(role)
                    ? "У тимлидера есть подчиненные"
                    : "У начальника отдела есть подчиненные";
            return blockUserAndSendMail(user, errorText, role);
        }
    }

    private IdmResponse removeManagerRole(User user, IdmRole role) {
        List<Long> clientIds = ppcRbac.getClientIdsByPrimaryManagerUid(user.getUid());
        List<Long> agencyIds = ppcRbac.getAgencyIdsByManagerUid(user.getUid());
        List<Long> campaignIds = ppcRbac.getCampaignIdsByManagerUid(user.getUid());

        if (!agencyIds.isEmpty() || !clientIds.isEmpty() || !campaignIds.isEmpty()) {
            String errorText = "У менеджера есть клиенты или агентства";
            return blockUserAndSendMail(user, errorText, role);
        }

        removeFromClientManagers(user.getUid());
        removeFromManagerHierarchy(user.getClientId());
        clientService.updateClientRole(user.getClientId(), RbacRole.EMPTY, null);

        logger.info("remove_role: 1, login: {}, role: {}", user.getLogin(), role);
        return new IdmSuccessResponse();
    }

    private IdmResponse removeLimitedSupportRole(User user, IdmRole role) {
        List<ClientsRelation> supportRelations = rbacClientsRelations.getSupportRelationsByOperatorIds(
                singletonList(user.getClientId()));

        if (!supportRelations.isEmpty()) {
            String errorText = "У саппорта есть клиенты";
            userService.blockUser(user.getClientId(), user.getUid());
            logger.info("block_role: 1, login: {}, role: {}, error_text: {}", user.getLogin(), role, errorText);
            return new IdmWarningResponse(errorText);
        }

        clientService.updateClientRole(user.getClientId(), RbacRole.EMPTY, null);
        logger.info("remove_role: 1, login: {}, role: {}", user.getLogin(), role);
        return new IdmSuccessResponse();
    }

    /**
     * Удаляет данные о менеджере из иерархии менеджеров
     */
    private void removeFromManagerHierarchy(ClientId managerClientId) {

        ManagerHierarchyInfo managerInfo = managerHierarchyService.getManagerData(managerClientId);
        if (managerInfo == null) {
            logger.error("Manager with ClientID {} is not found in manager_hierarchy", managerClientId);
            return;
        }
        checkState(managerInfo.getSubordinatesClientId().isEmpty(),
                "Manager with ClientID %s has subordinates", managerClientId);

        // исключаем менеджера из списков подчиненных для всех начальников вверх по иерархии
        List<ClientId> chiefsClientId = managerInfo.getChiefsClientId();
        if (!chiefsClientId.isEmpty()) {
            List<ManagerHierarchyInfo> chiefsInfo = managerHierarchyService.massGetManagerData(chiefsClientId);

            List<AppliedChanges<ManagerHierarchyInfo>> appliedChanges = StreamEx.of(chiefsInfo)
                    .map(chiefInfo -> removeManagerFromSubordinates(chiefInfo, managerInfo))
                    .toList();
            managerHierarchyService.updateManagerData(appliedChanges);
        }
        managerHierarchyService.deleteManagerData(managerClientId);
    }

    private static AppliedChanges<ManagerHierarchyInfo> removeManagerFromSubordinates(
            ManagerHierarchyInfo chiefInfo, ManagerHierarchyInfo managerInfo) {

        // исключаем менеджера из списков подчиненных прежнего руководителя
        List<ClientId> newSubordinatesClientId = StreamEx.of(chiefInfo.getSubordinatesClientId())
                .filter(clientId -> !Objects.equals(clientId, managerInfo.getManagerClientId()))
                .toList();
        List<Long> newSubordinatesUid = StreamEx.of(chiefInfo.getSubordinatesUid())
                .filter(uid -> !Objects.equals(uid, managerInfo.getManagerUid()))
                .toList();

        ModelChanges<ManagerHierarchyInfo> changes = new ModelChanges<>(chiefInfo.getId(), ManagerHierarchyInfo.class);

        changes.processNotNull(newSubordinatesClientId, ManagerHierarchyInfo.SUBORDINATES_CLIENT_ID);
        changes.processNotNull(newSubordinatesUid, ManagerHierarchyInfo.SUBORDINATES_UID);

        return changes.applyTo(chiefInfo);
    }

    private void removeFromClientManagers(Long managerUid) {
        List<Long> clientIds = clientService.getClientsOfManagerWithoutCampaigns(managerUid);
        clientService.deleteFromClientManagers(clientIds, managerUid);
    }

    private IdmWarningResponse blockUserAndSendMail(User user, String errorText, IdmRole role) {
        sendMail(user, errorText, role);
        userService.blockUser(user.getClientId(), user.getUid());
        logger.info("block_role: 1, login: {}, role: {}, error_text: {}", user.getLogin(), role, errorText);
        return new IdmWarningResponse(errorText);
    }

    private void sendMail(User user, String errorText, IdmRole role) {

        RemoveRoleEmailParameters parameters = new RemoveRoleEmailParameters.Builder()
                .setManagerLogin(user.getLogin())
                .setDomainLogin(user.getDomainLogin())
                .setRoleName(role.getName())
                .setErrorText(errorText)
                .build();

        mailSenderService.sendRemoveRoleEmail(parameters);
    }
}
