package ru.yandex.direct.intapi.entity.idm.service;

import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;

import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.intapi.entity.idm.model.IdmErrorResponse;
import ru.yandex.direct.intapi.entity.idm.model.IdmException;
import ru.yandex.direct.intapi.entity.idm.model.IdmFatalResponse;
import ru.yandex.direct.intapi.entity.idm.model.IdmResponse;
import ru.yandex.direct.intapi.entity.idm.model.IdmRole;
import ru.yandex.direct.intapi.entity.idm.model.RoleWithTeamleaders;

import static ru.yandex.direct.core.entity.user.utils.UserUtil.isDeveloper;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isSuperTeamLeader;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isTeamLeader;
import static ru.yandex.direct.utils.JsonUtils.toJson;

@ParametersAreNonnullByDefault
public class IdmRolesUtils {
    private IdmRolesUtils() {
    }

    static IdmResponse processIdmRequestSafe(Supplier<IdmResponse> idmResponseSupplier, Logger logger) {
        IdmResponse result;
        try {
            result = idmResponseSupplier.get();

        } catch (IdmException e) {
            logger.warn("error when processing request", e);
            String errorText = e.getErrorText();
            result = (e.getErrorType() == IdmException.IdmErrorType.FATAL)
                    ? new IdmFatalResponse(errorText)
                    : new IdmErrorResponse(errorText);

        } catch (RuntimeException e) {
            logger.error("error when processing request", e);
            result = new IdmErrorResponse("internal error");
        }
        logger.info("response: {}", toJson(result));
        return result;
    }

    static boolean isRoleAnyTeamLeader(IdmRole idmRole) {
        return IdmRole.TEAMLEADER.equals(idmRole) || IdmRole.SUPERTEAMLEADER.equals(idmRole);
    }

    static boolean isRoleDeveloper(IdmRole idmRole) {
        return IdmRole.DEVELOPER.equals(idmRole);
    }

    static boolean isRoleSuperReader(IdmRole idmRole) {
        return IdmRole.SUPERREADER.equals(idmRole);
    }

    static boolean isRoleAnySuperReader(IdmRole idmRole) {
        return isRoleDeveloper(idmRole) || isRoleSuperReader(idmRole);
    }

    static RoleWithTeamleaders getRoleWithTeamleaders(User user) {
        if (user.getRole() == null) {
            return RoleWithTeamleaders.EMPTY;
        } else if (isSuperTeamLeader(user)) {
            return RoleWithTeamleaders.SUPERTEAMLEADER;
        } else if (isTeamLeader(user)) {
            return RoleWithTeamleaders.TEAMLEADER;
        } else if (isDeveloper(user)) {
            return RoleWithTeamleaders.DEVELOPER;
        }
        return RoleWithTeamleaders.fromRbacRole(user.getRole());
    }
}
