package ru.yandex.direct.intapi.entity.idm.service;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.intapi.entity.idm.model.AddRoleRequest;
import ru.yandex.direct.intapi.entity.idm.model.IdmFatalResponse;
import ru.yandex.direct.intapi.entity.idm.model.IdmResponse;
import ru.yandex.direct.intapi.entity.idm.model.IdmSuccessResponse;
import ru.yandex.direct.intapi.entity.idm.model.IdmSupportForClientRole;
import ru.yandex.direct.intapi.entity.idm.model.RemoveRoleRequest;
import ru.yandex.direct.rbac.RbacClientsRelations;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.rbac.model.SupportClientRelation;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.intapi.entity.idm.model.AddRoleResponse.addRoleResponse;
import static ru.yandex.direct.intapi.entity.idm.service.IdmGetRolesService.defaultLoginIfEmpty;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.JsonUtils.toJson;

@Service
@ParametersAreNonnullByDefault
public class IdmSupportForClientService {
    private static final Logger logger = LoggerFactory.getLogger(IdmSupportForClientService.class);

    private static final Set<RbacRole> SUBJECT_CLIENT_REQUIRED_ROLES = Set.of(RbacRole.AGENCY, RbacRole.CLIENT);
    private static final Set<RbacRole> OPERATOR_REQUIRED_ROLES = Set.of(RbacRole.LIMITED_SUPPORT);

    private final UserService userService;
    private final RbacClientsRelations rbacClientsRelations;
    private final RbacService rbacService;
    private final ClientService clientService;
    private final IdmGetRolesService idmGetRolesService;


    @Autowired
    public IdmSupportForClientService(
            UserService userService,
            RbacClientsRelations rbacClientsRelations,
            RbacService rbacService,
            ClientService clientService,
            IdmGetRolesService idmGetRolesService) {
        this.userService = userService;
        this.rbacClientsRelations = rbacClientsRelations;
        this.rbacService = rbacService;
        this.clientService = clientService;
        this.idmGetRolesService = idmGetRolesService;
    }

    public IdmResponse addRole(AddRoleRequest request) {
        logger.info("addRole request: {}", toJson(request));

        Client subjectClient = Optional.ofNullable(request.getClientId())
                .map(ClientId::fromLong)
                .map(clientService::getClient)
                .orElse(null);

        User operator = Optional.ofNullable(request.getPassportLogin()).map(userService::getUserByLogin).orElse(null);

        String validationErrorText = validateAddRoleRequest(request, subjectClient, operator);
        if (validationErrorText != null) {
            logger.info("addRole: 0, validation error: {}", validationErrorText);
            return new IdmFatalResponse(validationErrorText);
        }

        // сейчас может быть запись c таким (client_id_from, client_id_to) но другим типом,
        // в этом случае упадем в insert
        rbacClientsRelations.addSupportRelation(ClientId.fromLong(subjectClient.getClientId()),
                operator.getClientId());

        logger.info("addRole: 1, login: {}, client_id: {}", request.getPassportLogin(), request.getClientId());
        return addRoleResponse(request.getPassportLogin(), request.getClientId());
    }

    public IdmResponse removeRole(RemoveRoleRequest request) {
        logger.info("remove_role request: {}", toJson(request));

        Client subjectClient = Optional.ofNullable(request.getClientId())
                .map(ClientId::fromLong)
                .map(clientService::getClient)
                .orElse(null);

        User operator = Optional.ofNullable(request.getPassportLogin()).map(userService::getUserByLogin).orElse(null);

        String validationErrorText = validateRemoveRoleRequest(request);
        if (validationErrorText != null) {
            logger.info("remove_role: 0, validation error: {}", validationErrorText);
            return new IdmFatalResponse(validationErrorText);
        }
        if (subjectClient == null) {
            logger.info("remove_role: 0, client not found for client_id {}", request.getClientId());
            return new IdmSuccessResponse();
        }
        if (operator == null) {
            logger.info("remove_role: 0, operator not found for passport-login {}", request.getPassportLogin());
            return new IdmFatalResponse(String.format("operator not found for passport-login %s",
                    request.getPassportLogin()));
        }

        rbacClientsRelations.removeSupportRelation(ClientId.fromLong(subjectClient.getClientId()),
                operator.getClientId());

        logger.info("remove_role: 1, login: {}, client_id: {}", request.getPassportLogin(), request.getClientId());
        return new IdmSuccessResponse();
    }

    private String validateAddRoleRequest(AddRoleRequest request, @Nullable Client subjectClient,
                                          @Nullable User operator) {
        // client_id
        if (request.getClientId() == null) {
            return "client_id is not defined";
        }
        if (subjectClient == null) {
            return String.format("client not found for client_id %s", request.getClientId());
        }
        if (!isValidRole(subjectClient.getRole(), SUBJECT_CLIENT_REQUIRED_ROLES)) {
            return String.format("нельзя получить доступ к пользователю с ролью %s", subjectClient.getRole());
        }

        // passport-login
        if (request.getPassportLogin() == null) {
            return "passport-login is not defined";
        }
        if (operator == null) {
            return String.format("operator not found for passport-login %s", request.getPassportLogin());
        }
        if (!isValidRole(operator.getRole(), OPERATOR_REQUIRED_ROLES)) {
            return String.format("operator with passport-login %s has wrong role", request.getPassportLogin());
        }
        return null;
    }

    private String validateRemoveRoleRequest(RemoveRoleRequest request) {
        if (request.getClientId() == null) {
            return "client_id is not defined";
        }
        if (request.getPassportLogin() == null) {
            return "passport-login is not defined";
        }
        return null;
    }

    private static boolean isValidRole(@Nullable RbacRole role, Set<RbacRole> validRoles) {
        return (role != null) && validRoles.contains(role);
    }

    public List<IdmSupportForClientRole> getAllRoles() {
        List<SupportClientRelation> relations = rbacClientsRelations.getAllSupportRelations();
        return convertToSupportForClientRoles(relations);
    }

    public List<IdmSupportForClientRole> getNextPageRoles(@Nullable SupportClientRelation lastReturnedRole,
                                                          int pageSize) {
        List<SupportClientRelation> relations =
                rbacClientsRelations.getNextPageSupportRelations(lastReturnedRole, pageSize);
        List<IdmSupportForClientRole> roles = convertToSupportForClientRoles(relations);
        // фильтровать при конвертации нельзя, так как нужно вернуть pageSize ролей
        checkState(roles.size() == relations.size());
        return roles;
    }

    private List<IdmSupportForClientRole> convertToSupportForClientRoles(List<SupportClientRelation> relations) {
        Set<ClientId> supportClientIds = listToSet(relations, SupportClientRelation::getSupportClientId);
        Map<ClientId, User> supportsByClientIds = getChiefUsersByClientIds(supportClientIds);

        return StreamEx.of(relations)
                .map(r -> {
                    User support = supportsByClientIds.get(r.getSupportClientId());
                    String domainLogin = defaultLoginIfEmpty(ifNotNull(support, User::getDomainLogin));
                    String passportLogin = defaultLoginIfEmpty(ifNotNull(support, User::getLogin));
                    return new IdmSupportForClientRole()
                            .withDomainLogin(domainLogin)
                            .withPassportLogin(passportLogin)
                            .withSupportClientId(r.getSupportClientId())
                            .withSubjectClientId(r.getSubjectClientId());
                })
                .toList();
    }

    private Map<ClientId, User> getChiefUsersByClientIds(Collection<ClientId> clientIds) {
        Map<ClientId, Long> chiefUidsByClientIds = rbacService.getChiefsByClientIds(clientIds);

        Collection<User> users = userService.massGetUser(chiefUidsByClientIds.values());
        Map<Long, User> usersByUids = listToMap(users, User::getUid);

        return EntryStream.of(chiefUidsByClientIds)
                .mapValues(usersByUids::get)
                .nonNullValues()
                .toMap();
    }
}
