package ru.yandex.direct.intapi.entity.idm.service;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.model.ClientPrimaryManager;
import ru.yandex.direct.intapi.entity.idm.model.AddRoleRequest;
import ru.yandex.direct.intapi.entity.idm.model.GetAllRolesResponse;
import ru.yandex.direct.intapi.entity.idm.model.GetAllRolesResponseGroupsItem;
import ru.yandex.direct.intapi.entity.idm.model.GetAllRolesResponseItem;
import ru.yandex.direct.intapi.entity.idm.model.IdmClientGroup;
import ru.yandex.direct.intapi.entity.idm.model.IdmFatalResponse;
import ru.yandex.direct.intapi.entity.idm.model.IdmGroupRoleName;
import ru.yandex.direct.intapi.entity.idm.model.IdmResponse;
import ru.yandex.direct.intapi.entity.idm.model.IdmRole;
import ru.yandex.direct.intapi.entity.idm.model.IdmSupportForClientRole;
import ru.yandex.direct.intapi.entity.idm.model.IdmUserRole;
import ru.yandex.direct.intapi.entity.idm.model.InfoResponse;
import ru.yandex.direct.intapi.entity.idm.model.InfoResponseField;
import ru.yandex.direct.intapi.entity.idm.model.InfoResponseRoleWithFields;
import ru.yandex.direct.intapi.entity.idm.model.InfoResponseRoles;
import ru.yandex.direct.intapi.entity.idm.model.RemoveRoleRequest;

import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.intapi.entity.idm.converter.GetAllRolesResponseConverter.convertBasicRolesForOneLogin;
import static ru.yandex.direct.intapi.entity.idm.converter.GetAllRolesResponseConverter.convertGroupRoles;
import static ru.yandex.direct.intapi.entity.idm.converter.GetAllRolesResponseConverter.convertUserRoles;
import static ru.yandex.direct.intapi.entity.idm.converter.IdmCommonNames.CLIENT_ID_SLUG;
import static ru.yandex.direct.intapi.entity.idm.converter.IdmCommonNames.MAIN_MANAGER_ROLE_SLUG_NAME;
import static ru.yandex.direct.intapi.entity.idm.converter.IdmCommonNames.PASSPORT_LOGIN;
import static ru.yandex.direct.intapi.entity.idm.converter.IdmCommonNames.SLUG;
import static ru.yandex.direct.intapi.entity.idm.converter.IdmCommonNames.SUPPORT_FOR_CLIENT_ROLE_NAME;

@Service
@ParametersAreNonnullByDefault
public class IdmUserManagementIntapiService {

    private static final String MAIN_MANAGER_ROLE_DESCRIPTION = "главный менеджер клиента";
    private static final String SUPPORT_FOR_CLIENT_ROLE_DESCRIPTION = "саппорт с доступом к клиенту";
    private static final String CLIENT_ID_DESCRIPTION = "ClientID клиента";
    private static final String PASSPORT_LOGIN_DESCRIPTION = "Паспортный логин";
    private static final String INFO_FIELD_TYPE_CHARFIELD = "charfield";
    private static final String INFO_FIELD_TYPE_UNDO = "undo";

    private final IdmGetRolesService idmGetRolesService;
    private final IdmAddRoleService idmAddRoleService;
    private final IdmRemoveRoleService idmRemoveRoleService;
    private final IdmGroupRolesService idmGroupRolesService;
    private final IdmMainManagerService idmMainManagerService;
    private final IdmSupportForClientService idmSupportForClientService;

    @Autowired
    public IdmUserManagementIntapiService(IdmGetRolesService idmGetRolesService,
                                          IdmAddRoleService idmAddRoleService,
                                          IdmRemoveRoleService idmRemoveRoleService,
                                          IdmGroupRolesService idmGroupRolesService,
                                          IdmMainManagerService idmMainManagerService,
                                          IdmSupportForClientService idmSupportForClientService) {
        this.idmGetRolesService = idmGetRolesService;
        this.idmAddRoleService = idmAddRoleService;
        this.idmRemoveRoleService = idmRemoveRoleService;
        this.idmGroupRolesService = idmGroupRolesService;
        this.idmMainManagerService = idmMainManagerService;
        this.idmSupportForClientService = idmSupportForClientService;
    }

    public GetAllRolesResponse getAllRoles() {
        List<IdmUserRole> basicRoles = idmGetRolesService.getAllRoles();
        List<ClientPrimaryManager> mainManagerRoles = idmMainManagerService.getAllRoles();
        List<IdmSupportForClientRole> supportForClientRoles = idmSupportForClientService.getAllRoles();

        List<GetAllRolesResponseItem> userItems =
                convertUserRoles(basicRoles, mainManagerRoles, supportForClientRoles);

        List<IdmClientGroup> groupRoles = idmGroupRolesService.getAllGroupRoles();
        List<GetAllRolesResponseGroupsItem> groupItems = convertGroupRoles(groupRoles);

        return new GetAllRolesResponse()
                .withUsers(userItems)
                .withGroups(groupItems);
    }

    public IdmResponse getUserBasicRoles(@Nullable String domainLogin) {
        if (StringUtils.isEmpty(domainLogin)) {
            return new IdmFatalResponse("domain login required");
        }
        List<IdmUserRole> userRoles = idmGetRolesService.getUserRoles(domainLogin);
        return convertBasicRolesForOneLogin(userRoles);
    }

    public InfoResponse info() {
        InfoResponseField clientIdField = new InfoResponseField()
                .withSlug(CLIENT_ID_SLUG)
                .withName(CLIENT_ID_DESCRIPTION)
                .withRequired(true)
                .withType(INFO_FIELD_TYPE_CHARFIELD);

        InfoResponseField passportLoginField = new InfoResponseField()
                .withSlug(PASSPORT_LOGIN)
                .withName(PASSPORT_LOGIN_DESCRIPTION)
                .withRequired(true);
        InfoResponseField passportLoginUndoField = new InfoResponseField()
                .withSlug(PASSPORT_LOGIN)
                .withType(INFO_FIELD_TYPE_UNDO);

        InfoResponseRoleWithFields managerForClientRole = new InfoResponseRoleWithFields()
                .withName(IdmGroupRoleName.MANAGER_FOR_CLIENT.getName())
                .withFields(asList(clientIdField, passportLoginUndoField));

        InfoResponseRoleWithFields mainManagerForClientRole = new InfoResponseRoleWithFields()
                .withName(MAIN_MANAGER_ROLE_DESCRIPTION)
                .withFields(singletonList(clientIdField))
                .withIsExclusive(true);

        InfoResponseRoleWithFields supportForClientRole = new InfoResponseRoleWithFields()
                .withName(SUPPORT_FOR_CLIENT_ROLE_DESCRIPTION)
                .withFields(singletonList(clientIdField));

        Map<String, Object> roleValues = new HashMap<>();
        roleValues.put(IdmGroupRoleName.MANAGER_FOR_CLIENT.getTypedValue(), managerForClientRole);
        roleValues.put(MAIN_MANAGER_ROLE_SLUG_NAME, mainManagerForClientRole);
        roleValues.put(SUPPORT_FOR_CLIENT_ROLE_NAME, supportForClientRole);

        Set<IdmRole> enabledBasicRoleNames = idmGetRolesService.getEnabledRoleNames();
        Map<String, String> basicRoles = StreamEx.of(enabledBasicRoleNames)
                .mapToEntry(IdmRole::getTypedValue, IdmRole::getName)
                .toMap();
        roleValues.putAll(basicRoles);

        return new InfoResponse()
                .withRoles(new InfoResponseRoles()
                        .withSlug(SLUG)
                        .withName("роль")
                        .withValues(roleValues))
                .withFields(singletonList(passportLoginField));
    }

    public IdmResponse addRole(AddRoleRequest request) {
        if (IdmGroupRoleName.MANAGER_FOR_CLIENT.getTypedValue().equalsIgnoreCase(request.getRole())) {
            return idmGroupRolesService.addGroupRole(request);
        }
        if (MAIN_MANAGER_ROLE_SLUG_NAME.equalsIgnoreCase(request.getRole())) {
            return idmMainManagerService.addRole(request);
        } else if (SUPPORT_FOR_CLIENT_ROLE_NAME.equalsIgnoreCase(request.getRole())) {
            return idmSupportForClientService.addRole(request);
        } else {
            return idmAddRoleService.addRole(request);
        }
    }

    public IdmResponse removeRole(RemoveRoleRequest request) {
        if (IdmGroupRoleName.MANAGER_FOR_CLIENT.getTypedValue().equalsIgnoreCase(request.getRole())) {
            return idmGroupRolesService.removeGroupRole(request);
        }
        if (MAIN_MANAGER_ROLE_SLUG_NAME.equalsIgnoreCase(request.getRole())) {
            return idmMainManagerService.removeRole(request);
        } else if (SUPPORT_FOR_CLIENT_ROLE_NAME.equalsIgnoreCase(request.getRole())) {
            return idmSupportForClientService.removeRole(request);
        } else {
            return idmRemoveRoleService.removeRole(request);
        }
    }
}
