package ru.yandex.direct.intapi.entity.internalads.controller;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import ru.yandex.direct.core.entity.banner.model.BannerWithInternalInfo;
import ru.yandex.direct.core.entity.banner.model.TemplateVariable;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.image.model.BannerImageFormat;
import ru.yandex.direct.core.entity.image.repository.BannerImageFormatRepository;
import ru.yandex.direct.core.entity.image.service.ImageUtils;
import ru.yandex.direct.core.entity.internalads.service.TemplateResourceService;
import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.entity.internalads.model.InternalAdResponse;
import ru.yandex.direct.intapi.entity.internalads.model.TemplateVariableResponse;
import ru.yandex.direct.tvm.AllowServices;

import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_TEST;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Ручки для получения информации о баннерах внутренней рекламы
 */
@ParametersAreNonnullByDefault
@Controller
@Api(value = "API для получения информации о внутренней рекламы")
@RequestMapping(path = "/internalads")
@AllowServices(production = {DIRECT_WEB_PROD}, testing = {DIRECT_WEB_TEST, DIRECT_DEVELOPER})
public class InternalAdsController {
    private final TemplateResourceService templateResourceService;
    private final BannerImageFormatRepository bannerImageFormatRepository;
    private final BannerTypedRepository bannerTypedRepository;

    @Autowired
    public InternalAdsController(TemplateResourceService templateResourceService,
                                 BannerImageFormatRepository bannerImageFormatRepository,
                                 BannerTypedRepository bannerTypedRepository) {
        this.templateResourceService = templateResourceService;
        this.bannerImageFormatRepository = bannerImageFormatRepository;
        this.bannerTypedRepository = bannerTypedRepository;
    }

    @ApiOperation(
            value = "internal_ad_info",
            nickname = "internal_ad_info",
            httpMethod = "POST"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = ErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = Object.class)
            }
    )
    @RequestMapping(path = "/internal_ad_info", method = RequestMethod.POST)
    @ResponseBody
    public Map<Long, InternalAdResponse> getInternalAdInfoByBannerId(
            @RequestParam(value = "shard") int shard,
            @RequestBody List<Long> bannerIds) {
        List<BannerWithInternalInfo> banners = bannerTypedRepository.getSafely(shard, bannerIds,
                BannerWithInternalInfo.class);
        Map<Long, BannerWithInternalInfo> internalExtraInfoByAdId = listToMap(banners,
                BannerWithInternalInfo::getId);
        Set<Long> imageTemplateResourceIds = collectImageResourceIds(internalExtraInfoByAdId);
        Set<String> imageHashes = collectImageHashes(internalExtraInfoByAdId, imageTemplateResourceIds);
        Map<String, String> imageHashToImageUrlMap = calculateImageUrls(shard, imageHashes);
        return prepareResponse(internalExtraInfoByAdId, imageTemplateResourceIds, imageHashToImageUrlMap);
    }

    private static InternalAdResponse convertInternalAdExtraInfo(BannerWithInternalInfo info) {
        List<TemplateVariableResponse> result = mapList(info.getTemplateVariables(),
                InternalAdsController::convertTemplateVariableResponse);
        return new InternalAdResponse(info.getDescription(), info.getTemplateId(), result);
    }

    private static TemplateVariableResponse convertTemplateVariableResponse(TemplateVariable templateVariable) {
        return new TemplateVariableResponse(templateVariable.getTemplateResourceId(),
                templateVariable.getInternalValue());
    }

    private Set<Long> collectImageResourceIds(Map<Long, BannerWithInternalInfo> internalExtraInfoByAdId) {
        Set<Long> templateIds = listToSet(internalExtraInfoByAdId.values(), BannerWithInternalInfo::getTemplateId);
        return templateResourceService.getImageResourceIdsByTemplateIds(templateIds);
    }

    private Set<String> collectImageHashes(Map<Long, BannerWithInternalInfo> internalExtraInfoByAdId,
                                           Set<Long> imageTemplateResourceIds) {
        List<TemplateVariable> templateVariables;
        Set<String> imageHashes = new HashSet<>();
        // собираем все imageHashes
        for (Map.Entry<Long, BannerWithInternalInfo> entry : internalExtraInfoByAdId.entrySet()) {
            templateVariables = entry.getValue().getTemplateVariables();
            for (TemplateVariable variable : templateVariables) {
                if (imageTemplateResourceIds.contains(variable.getTemplateResourceId())
                        && variable.getInternalValue() != null) {
                    imageHashes.add(variable.getInternalValue());
                }
            }
        }
        return imageHashes;
    }

    private Map<String, String> calculateImageUrls(int shard, Set<String> imageHashes) {
        // создаем map imageHash + imageUrl
        Map<String, BannerImageFormat> bannerImageFormats =
                bannerImageFormatRepository.getBannerImageFormats(shard, imageHashes);
        Map<String, String> imageHashToImageUrlMap = new HashMap<>();
        for (Map.Entry<String, BannerImageFormat> formats : bannerImageFormats.entrySet()) {
            imageHashToImageUrlMap.put(formats.getKey(), ImageUtils.generateOrigImageUrl(formats.getValue()));
        }
        return imageHashToImageUrlMap;
    }

    private Map<Long, InternalAdResponse> prepareResponse(Map<Long, BannerWithInternalInfo> internalExtraInfoByAdId,
                                                          Set<Long> imageTemplateResourceIds,
                                                          Map<String, String> imageHashToImageUrlMap) {
        // заменяем imageHash на imageUrl для баннеров внутренней рекламы
        String imageHash;
        List<TemplateVariable> templateVariables;
        Map<Long, InternalAdResponse> resultInternalAdMap = new HashMap<>();
        for (Map.Entry<Long, BannerWithInternalInfo> entry : internalExtraInfoByAdId.entrySet()) {
            templateVariables = entry.getValue().getTemplateVariables();
            for (TemplateVariable variable : templateVariables) {
                if (imageTemplateResourceIds.contains(variable.getTemplateResourceId())) {
                    if (variable.getInternalValue() != null
                            && imageHashToImageUrlMap.containsKey(variable.getInternalValue())) {
                        imageHash = variable.getInternalValue();
                        variable.setInternalValue(imageHashToImageUrlMap.get(imageHash));
                    }
                }
            }
            entry.getValue().setTemplateVariables(templateVariables);
            resultInternalAdMap.put(entry.getKey(), convertInternalAdExtraInfo(entry.getValue()));
        }
        return resultInternalAdMap;
    }
}
