package ru.yandex.direct.intapi.entity.internalads.controller;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import ru.yandex.direct.core.entity.internalads.model.InternalAdPlaceInfo;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsOperatorProductAccess;
import ru.yandex.direct.core.entity.internalads.service.InternalAdsOperatorProductAccessService;
import ru.yandex.direct.core.entity.internalads.service.PlaceService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.entity.internalads.model.CanControlPlaceResponse;
import ru.yandex.direct.intapi.entity.internalads.model.PlaceListResponse;
import ru.yandex.direct.intapi.entity.internalads.model.PlaceResponse;
import ru.yandex.direct.tvm.AllowServices;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_TEST;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
@Controller
@Api(value = "API для получения плейсов внутренней рекламы")
@RequestMapping(path = "/internalads/places")
@AllowServices(production = {DIRECT_WEB_PROD}, testing = {DIRECT_WEB_TEST, DIRECT_DEVELOPER})
public class PlacesController {
    private final InternalAdsOperatorProductAccessService internalAdsOperatorProductAccessService;
    private final PlaceService placeService;

    @Autowired
    public PlacesController(InternalAdsOperatorProductAccessService internalAdsOperatorProductAccessService,
                            PlaceService placeService) {
        this.internalAdsOperatorProductAccessService = internalAdsOperatorProductAccessService;
        this.placeService = placeService;
    }

    @ApiOperation(
            value = "controllable_places",
            nickname = "controllable_places",
            httpMethod = "GET"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = ErrorResponse.class),
                    @ApiResponse(code = 403, message = "permission denied", response = ErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = PlaceListResponse.class)
            }
    )
    @RequestMapping(path = "/controllable_places", method = RequestMethod.GET)
    @ResponseBody
    public PlaceListResponse getControllablePlaces(
            @RequestParam(value = "operator_uid") Long operatorUid,
            @RequestParam(value = "product_client_id") Long longProductClientId) {
        ClientId productClientId = ClientId.fromLong(longProductClientId);

        InternalAdsOperatorProductAccess access = internalAdsOperatorProductAccessService.getAccess(
                operatorUid, productClientId);

        Set<Long> placeIds = access.placesWithWriteAccess();
        if (placeIds.isEmpty()) {
            return accessToNoPlaces(operatorUid, productClientId.asLong());
        }

        Map<Long, InternalAdPlaceInfo> placeInfoMap = listToMap(
                placeService.getPlaceInfoForValidPlaces(), InternalAdPlaceInfo::getId);

        List<PlaceResponse> result = mapList(placeIds, placeId -> convertPlace(placeInfoMap.get(placeId)));
        return new PlaceListResponse(operatorUid, productClientId.asLong(), checkNotNull(result));
    }

    @ApiOperation(
            value = "can_control_place",
            nickname = "can_control_place",
            httpMethod = "GET"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = ErrorResponse.class),
                    @ApiResponse(code = 403, message = "permission denied", response = ErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = CanControlPlaceResponse.class)
            }
    )
    @RequestMapping(path = "/can_control_place", method = RequestMethod.GET)
    @ResponseBody
    public CanControlPlaceResponse canControlPlace(
            @RequestParam(value = "operator_uid") Long operatorUid,
            @RequestParam(value = "product_client_id") Long longProductClientId,
            @RequestParam(value = "place_id") Long placeId) {
        ClientId productClientId = ClientId.fromLong(longProductClientId);
        InternalAdsOperatorProductAccess access = internalAdsOperatorProductAccessService.getAccess(
                operatorUid, productClientId);

        return new CanControlPlaceResponse(operatorUid, productClientId.asLong(), placeId,
                access.hasWriteAccessToPlace(placeId));
    }

    private PlaceListResponse accessToNoPlaces(Long operatorUid, Long productClientId) {
        return new PlaceListResponse(operatorUid, productClientId, Collections.emptyList());
    }

    private static PlaceResponse convertPlace(InternalAdPlaceInfo info) {
        return new PlaceResponse(info.getId(), info.getFullDescription());
    }

}
