package ru.yandex.direct.intapi.entity.inventori.controller;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nonnull;
import javax.ws.rs.core.MediaType;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.core.entity.inventori.service.InventoriServiceCore;
import ru.yandex.direct.intapi.entity.inventori.model.InventoriResponse;
import ru.yandex.direct.intapi.entity.inventori.model.InventoriResult;
import ru.yandex.direct.intapi.entity.inventori.service.InventoriValidationService;
import ru.yandex.direct.inventori.model.request.CampaignPredictionRequest;
import ru.yandex.direct.tvm.AllowServices;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.INVENTORI_PROD;
import static ru.yandex.direct.tvm.TvmService.INVENTORI_TEST;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@RestController
@Api(value = "API для получения CampaignPrediction запроса в InventORI в формате JSON")
@RequestMapping(path = "/inventori")
@AllowServices(production = {INVENTORI_PROD}, testing = {INVENTORI_TEST, DIRECT_DEVELOPER})
public class InventoriController {

    private static final Logger logger = LoggerFactory.getLogger(InventoriController.class);
    private final InventoriServiceCore inventoriServiceCore;
    private final InventoriValidationService inventoriValidationService;

    @Autowired
    public InventoriController(InventoriServiceCore inventoriServiceCore,
                               InventoriValidationService inventoriValidationService) {
        this.inventoriServiceCore = inventoriServiceCore;
        this.inventoriValidationService = inventoriValidationService;
    }

    @ApiOperation(
            value = "Получение CampaignPrediction запроса в InventORI в формате JSON",
            nickname = "/campaigns_info"
    )
    @RequestMapping(
            path = "/campaigns_info",
            method = RequestMethod.GET,
            produces = MediaType.APPLICATION_JSON)
    public InventoriResponse getInventoriRequests(
            @ApiParam(value = "ID кампаний")
            @RequestParam(value = "campaign_ids") String data) {
        List<Long> campaignIds = StreamEx.of(data.split(","))
                .map(Long::parseLong)
                .distinct()
                .toImmutableList();

        return getInventoriResponse(campaignIds);
    }

    @ApiOperation(
            value = "Ручка со списком подтверждённых прайсовых кампаний на главной",
            nickname = "/booked_yndxfrontpage_campaigns"
    )
    @RequestMapping(
            path = "/booked_yndxfrontpage_campaigns",
            method = RequestMethod.GET,
            produces = MediaType.APPLICATION_JSON)
    public List<Long> getBookedYndxFrontpageCampaigns() {
        List<Long> campaignIds = inventoriServiceCore.getBookedYndxFrontpageCampaigns();
        return campaignIds;
    }

    @ApiOperation(
            value = "Ручка со списком подтверждённых прайсовых кампаний на главной",
            nickname = "/booked_campaigns"
    )
    @RequestMapping(
            path = "/booked_campaigns",
            method = RequestMethod.GET,
            produces = MediaType.APPLICATION_JSON)
    public List<Long> getBookedCampaigns() {
        List<Long> campaignIds = inventoriServiceCore.getBookedCampaigns();
        return campaignIds;
    }

    @RequestMapping(
            path = "/campaigns_info",
            method = RequestMethod.POST,
            produces = MediaType.APPLICATION_JSON)
    public InventoriResponse getInventoriRequestsWithBody(
            @ApiParam(value = "ID кампаний")
            @RequestBody @Nonnull CampaignIdsList data) {
        List<Long> campaignIds = StreamEx.of(data.getCampaignIds())
                .map(Long::parseLong)
                .distinct()
                .toImmutableList();
        return getInventoriResponse(campaignIds);
    }
    private InventoriResponse getInventoriResponse(List<Long> campaignIds) {
        ValidationResult<List<Long>, Defect> validationResult =
                inventoriValidationService.validateCampaignIds(campaignIds);
        List<Long> validCampaignIds = ValidationResult.getValidItems(validationResult);
        List<InventoriResult> results = new ArrayList<>(campaignIds.size());
        try {
            List<CampaignPredictionRequest> campaignPredictionRequest =
                    inventoriServiceCore.getCampaignPredictionRequestForCampaigns(
                            true,
                            null,
                            null,
                            validCampaignIds,
                            false,
                            true,
                            true,
                            false,
                            validationResult);
            List<Long> validCampaignIdsAfterRequest = ValidationResult.getValidItems(validationResult);
            results.addAll(mapList(filterList(campaignPredictionRequest,
                    vr -> validCampaignIdsAfterRequest.contains(vr.getCampaignId())), InventoriResult::success));
        } catch (Exception e) {
            logger.info(String.format("Inventori campaigns_info error: {\"internal_direct_error\": \"%s\"}", e));
            return new InventoriResponse(e.toString());
        }
        results.addAll(inventoriValidationService.processValidationResultErrorsToResponse(validationResult));

        return new InventoriResponse(results);
    }

    public static class CampaignIdsList {
        private List<String>  campaignIds;
        public List<String> getCampaignIds() {
            return campaignIds;
        }

        public void setCampaignIds(List<String> campaignIds) {
            this.campaignIds = campaignIds;
        }
    }
}
