package ru.yandex.direct.intapi.entity.inventori.service;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.inventori.service.validation.InventoriDefectIds;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.intapi.entity.inventori.model.InventoriResult;
import ru.yandex.direct.intapi.entity.inventori.model.InventoriResultErrorIds;
import ru.yandex.direct.intapi.validation.kernel.TranslatableIntapiDefect;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.presentation.DefectPresentationRegistry;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectId;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.entity.inventori.validation.CampaignDefectIds;

import static ru.yandex.direct.intapi.entity.inventori.model.InventoriResultErrorIds.CAMPAIGN_ALREADY_DELETED;
import static ru.yandex.direct.intapi.entity.inventori.model.InventoriResultErrorIds.CAMPAIGN_NOT_EXISTS;
import static ru.yandex.direct.intapi.entity.inventori.model.InventoriResultErrorIds.ERROR_IDENTITY_PROBLEM;
import static ru.yandex.direct.intapi.entity.inventori.model.InventoriResultErrorIds.INVALID_CAMPAIGN_STRATEGY;
import static ru.yandex.direct.intapi.entity.inventori.model.InventoriResultErrorIds.INVALID_CAMPAIGN_TYPE;
import static ru.yandex.direct.intapi.entity.inventori.model.InventoriResultErrorIds.INVALID_GROUP_TYPE;
import static ru.yandex.direct.intapi.entity.inventori.model.InventoriResultErrorIds.UNEXPECTED_INVENTORI_COMMON_PROBLEM;
import static ru.yandex.direct.intapi.entity.inventori.model.InventoriResultErrorIds.UNEXPECTED_INVENTORI_PROBLEM_WITH_CAMPAIGN;
import static ru.yandex.direct.web.core.entity.inventori.validation.InventoriConstraints.campaignExists;
import static ru.yandex.direct.web.core.entity.inventori.validation.InventoriConstraints.validGroups;
import static ru.yandex.direct.web.core.entity.inventori.validation.InventoriConstraints.validStatusEmpty;
import static ru.yandex.direct.web.core.entity.inventori.validation.InventoriConstraints.validStrategy;

/**
 * Сервис для валидации запросов на запросы из InventORI.
 */
@ParametersAreNonnullByDefault
@Service
public class InventoriValidationService {

    private static final Logger logger = LoggerFactory.getLogger(InventoriValidationService.class);

    private final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;
    private final DefectPresentationRegistry<TranslatableIntapiDefect> defectPresentationRegistry;
    private final TranslationService translationService;
    private final AdGroupRepository adGroupRepository;

    @Autowired
    public InventoriValidationService(ShardHelper shardHelper, CampaignRepository campaignRepository,
                                      DefectPresentationRegistry<TranslatableIntapiDefect> defectPresentationRegistry,
                                      TranslationService translationService, AdGroupRepository adGroupRepository) {
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
        this.defectPresentationRegistry = defectPresentationRegistry;
        this.translationService = translationService;
        this.adGroupRepository = adGroupRepository;
    }

    public ValidationResult<List<Long>, Defect> validateCampaignIds(List<Long> campaignIds) {
        Map<Long, Campaign> campaignById = new HashMap<>(campaignIds.size());

        var shardToCampaignIds = shardHelper.groupByShard(campaignIds, ShardKey.CID);
        shardToCampaignIds.forEach((shard, cids) ->
                campaignById.putAll(StreamEx.of(campaignRepository.getCampaigns(shard, cids))
                        .mapToEntry(Campaign::getId, Function.identity())
                        .toMap()));

        Map<Long, List<AdGroupSimple>> adGroupsTypeByCampaignId = new HashMap<>();
        shardToCampaignIds.forEach((shard, cids) ->
                adGroupsTypeByCampaignId.putAll(adGroupRepository.getAdGroupSimpleByCampaignsIds(shard, cids)));

        Set<Long> existingCampaignIds = campaignById.keySet();

        ListValidationBuilder<Long, Defect> listValidationBuilder =
                ListValidationBuilder.of(campaignIds, Defect.class);

        listValidationBuilder
                .checkEach(campaignExists(existingCampaignIds))
                .checkEach(validStatusEmpty(campaignById), When.isValid())
                .checkEach(validStrategy(campaignById), When.isValid())
                .checkEach(validGroups(adGroupsTypeByCampaignId), When.isValid());

        return listValidationBuilder.getResult();
    }

    public List<InventoriResult> processValidationResultErrorsToResponse(ValidationResult<?, Defect> vr) {
        return StreamEx.of(vr.flattenErrors())
                .map(i -> {
                    InventoriResultErrorIds errorId = convertDefectInfoIntoInventoriResultErrorIds(i);
                    String errorText = convertDefectInfoToString(i);
                    logger.info(String.format("Inventori campaigns_info error: {\"error_id\": %d, \"error\": \"%s\"}",
                            errorId.getValue(), errorText));
                    return InventoriResult.error((Long) i.getValue(), errorId, errorText);
                })
                .toList();
    }

    private InventoriResultErrorIds convertDefectInfoIntoInventoriResultErrorIds(DefectInfo<Defect> defectInfo) {
        Map<DefectId, InventoriResultErrorIds> mapper = Map.of(
                InventoriDefectIds.String.UNEXPECTED_INVENTORI_PROBLEM_WITH_CAMPAIGN,
                UNEXPECTED_INVENTORI_PROBLEM_WITH_CAMPAIGN,
                InventoriDefectIds.String.UNEXPECTED_INVENTORI_COMMON_PROBLEM, UNEXPECTED_INVENTORI_COMMON_PROBLEM,
                CampaignDefectIds.CampaignDefects.CAMPAIGN_NOT_EXISTS, CAMPAIGN_NOT_EXISTS,
                CampaignDefectIds.CampaignDefects.CAMPAIGN_ALREADY_DELETED, CAMPAIGN_ALREADY_DELETED,
                CampaignDefectIds.StrategyDefects.INVALID_CAMPAIGN_STRATEGY, INVALID_CAMPAIGN_STRATEGY,
                CampaignDefectIds.CampaignTypeDefects.INVALID_CAMPAIGN_TYPE, INVALID_CAMPAIGN_TYPE,
                CampaignDefectIds.GroupsDefects.INVALID_GROUP_TYPE, INVALID_GROUP_TYPE
        );
        return mapper.getOrDefault(defectInfo.getDefect().defectId(), ERROR_IDENTITY_PROBLEM);
    }

    private String convertDefectInfoToString(DefectInfo<Defect> defectInfo) {
        return translationService.translate(defectPresentationRegistry.getPresentation(defectInfo).getTranslatable());
    }
}
