package ru.yandex.direct.intapi.entity.keywords.controller;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.annotation.Nonnull;
import javax.ws.rs.core.MediaType;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.IntApiException;
import ru.yandex.direct.intapi.entity.keywords.model.CampaignIdAndBannerIdPair;
import ru.yandex.direct.intapi.entity.keywords.model.GetKeywordsByCidAndBidPairResult;
import ru.yandex.direct.intapi.entity.keywords.service.BannerKeywordsService;

import static ru.yandex.direct.intapi.common.converter.directtsv.DirectTsvMessageConverter.DIRECT_TSV_MEDIA_TYPE;


@RestController
@Api(value = "Экспорт фраз, использующихся в баннерах Директа")
public class KeywordsController {
    static final String CID_BID_PAIR_PARAM_NAME = "data";
    private static final String NEW_LINE_PATTERN = "\n";
    private static final String TAB_PATTERN = "\t";

    private final BannerKeywordsService bannerKeywordsService;

    @Autowired
    public KeywordsController(BannerKeywordsService bannerKeywordsService) {
        this.bannerKeywordsService = bannerKeywordsService;
    }

    @ApiOperation(
            value = "Получение списка фраз по парам cid и bid",
            notes = "В результирующий список попадут только найденные в базе баннеры"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params",
                            response = ErrorResponse.class)
            }

    )
    @RequestMapping(path = "keywords/by-cid-pid-pairs",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON,
            produces = MediaType.APPLICATION_JSON
    )
    public List<GetKeywordsByCidAndBidPairResult> getKeywordsByCidAndBidPair(
            @ApiParam(value = "Список пар cid и bid, по которым получаем ключевые фразы", required = true)
            @RequestBody @Nonnull List<CampaignIdAndBannerIdPair> params) {
        return bannerKeywordsService.getKeywordsByCidAndBidPairs(params);
    }

    @ApiOperation(
            value = "Получение списка фраз по парам cid и bid",
            notes = "В результирующий список попадут только найденные в базе баннеры",
            nickname = "GetBannersPhrases"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params",
                            response = ErrorResponse.class)
            }
    )
    @RequestMapping(path = "GetBannersPhrases", method = RequestMethod.GET, produces = DIRECT_TSV_MEDIA_TYPE)
    public List<GetKeywordsByCidAndBidPairResult> getKeywordsByCidAndBidPairTsv(
            @ApiParam(value = "Список пар cid и bid, по которым получаем ключевые фразы в формате " +
                    "<cid1>\\t<bid1>\\n<cid2>\\t<bid2>\\n<...>")
            @RequestParam(value = CID_BID_PAIR_PARAM_NAME) String data) {
        return bannerKeywordsService.getKeywordsByCidAndBidPairs(parseRequestTsvData(data));
    }

    List<CampaignIdAndBannerIdPair> parseRequestTsvData(String data) {
        List<CampaignIdAndBannerIdPair> params = new ArrayList<>();
        for (String line : data.split(NEW_LINE_PATTERN)) {
            if (line.isEmpty() || line.startsWith("#")) {
                continue;
            }
            String[] values = line.split(TAB_PATTERN);
            if (values.length != 2) {
                throw new IntApiException(HttpStatus.BAD_REQUEST,
                        new ErrorResponse(ErrorResponse.ErrorCode.BAD_PARAM,
                                "Invalid data line " + line));
            }

            try {
                params.add(
                        new CampaignIdAndBannerIdPair()
                                .withCampaignId(Long.parseLong(values[0]))
                                .withBannerId(Long.parseLong(values[1])));
            } catch (NumberFormatException e) {
                throw new IntApiException(HttpStatus.BAD_REQUEST,
                        new ErrorResponse(ErrorResponse.ErrorCode.BAD_PARAM,
                                "Unparsable value in " + Arrays.asList(values)));
            }
        }
        return params;
    }
}
