package ru.yandex.direct.intapi.entity.keywords.repository;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Record4;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.intapi.entity.keywords.model.CampaignIdAndBannerIdPair;
import ru.yandex.direct.intapi.entity.keywords.model.GetKeywordsByCidAndBidPairResult;

import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Мини-репозиторий для работы с фразами для ручки
 * {@link ru.yandex.direct.intapi.entity.keywords.controller.KeywordsController}
 * <p>
 * Расположен здесь, потому что потенциально не может использоваться нигде, кроме этой ручки.
 */
@Repository
@ParametersAreNonnullByDefault
public class BannerKeywordsForIntapiRepository {
    private final DslContextProvider dslContextProvider;

    @Autowired
    public BannerKeywordsForIntapiRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Получить фразы по парам cid и bid в заданном шарде.
     *
     * @return список объектов, содержащих id кампании, id баннера, id фразы и текст фразы
     */
    public List<GetKeywordsByCidAndBidPairResult> getPhrases(int shard, List<CampaignIdAndBannerIdPair> filters) {
        return dslContextProvider.ppc(shard)
                .select(BIDS.ID, BIDS.PHRASE, BANNERS.CID, BANNERS.BID)
                .from(BANNERS)
                .innerJoin(BIDS).on(BIDS.PID.eq(BANNERS.PID))
                .where(row(BANNERS.CID, BANNERS.BID).in(mapList(filters, f -> row(f.getCampaignId(), f.getBannerId()))))
                .fetch(this::extractResult);
    }

    private GetKeywordsByCidAndBidPairResult extractResult(Record4<Long, String, Long, Long> x) {
        return new GetKeywordsByCidAndBidPairResult()
                .withCid(x.getValue(BANNERS.CID))
                .withBid(x.getValue(BANNERS.BID))
                .withKeywordId(x.getValue(BIDS.ID))
                .withKeyword(x.getValue(BIDS.PHRASE));
    }

}
