package ru.yandex.direct.intapi.entity.keywords.service;

import java.util.List;
import java.util.function.Function;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.intapi.entity.keywords.model.CampaignIdAndBannerIdPair;
import ru.yandex.direct.intapi.entity.keywords.model.GetKeywordsByCidAndBidPairResult;
import ru.yandex.direct.intapi.entity.keywords.repository.BannerKeywordsForIntapiRepository;
import ru.yandex.direct.intapi.validation.IntApiDefect;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.intapi.validation.IntApiConstraints.notEmptyCollection;
import static ru.yandex.direct.intapi.validation.IntApiConstraints.notNull;
import static ru.yandex.direct.intapi.validation.IntApiConstraints.validId;
import static ru.yandex.direct.intapi.validation.ValidationUtils.checkResult;

@Service
public class BannerKeywordsService {
    private final ShardHelper shardHelper;
    private final BannerKeywordsForIntapiRepository bannerKeywordsForIntapiRepository;

    @Autowired
    public BannerKeywordsService(ShardHelper shardHelper,
                                 BannerKeywordsForIntapiRepository bannerKeywordsForIntapiRepository) {
        this.shardHelper = shardHelper;
        this.bannerKeywordsForIntapiRepository = bannerKeywordsForIntapiRepository;
    }

    /**
     * Получить из всех шардов информацию о фразах по парам идентификаторов их кампании и баннера
     */
    public List<GetKeywordsByCidAndBidPairResult> getKeywordsByCidAndBidPairs(List<CampaignIdAndBannerIdPair> params) {
        ValidationResult<List<CampaignIdAndBannerIdPair>, IntApiDefect> validationResult = validate(params);
        checkResult(validationResult);

        return shardHelper.groupByShard(params, ShardKey.CID, CampaignIdAndBannerIdPair::getCampaignId)
                .chunkedBy(SqlUtils.TYPICAL_SELECT_CHUNK_SIZE)
                .stream()
                .map(el -> bannerKeywordsForIntapiRepository.getPhrases(el.getKey(), el.getValue()))
                .toFlatList(Function.identity());
    }

    public ValidationResult<List<CampaignIdAndBannerIdPair>, IntApiDefect> validate(
            List<CampaignIdAndBannerIdPair> params) {
        ListValidationBuilder<CampaignIdAndBannerIdPair, IntApiDefect> v =
                ListValidationBuilder.of(params, IntApiDefect.class);

        return v.check(notNull())
                .check(notEmptyCollection())
                .checkEach(notNull())
                .checkEachBy(this::requestPairIsValid, When.notNull())
                .getResult();
    }

    private ValidationResult<CampaignIdAndBannerIdPair, IntApiDefect> requestPairIsValid(
            CampaignIdAndBannerIdPair param) {
        ItemValidationBuilder<CampaignIdAndBannerIdPair, IntApiDefect> v =
                ItemValidationBuilder.of(param, IntApiDefect.class);

        v.item(param.getCampaignId(), CampaignIdAndBannerIdPair.CAMPAIGN_IDENTIFIER)
                .check(notNull())
                .check(validId());
        v.item(param.getBannerId(), CampaignIdAndBannerIdPair.BANNER_IDENTIFIER)
                .check(notNull())
                .check(validId());

        return v.getResult();
    }
}
