package ru.yandex.direct.intapi.entity.launcher.service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.container.AdGroupsSelectionCriteria;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupStatus;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.StatusModerate;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.banner.model.BannerWithHref;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.service.BannerService;
import ru.yandex.direct.core.entity.campaign.model.CampaignForLauncher;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.mobilecontent.model.AgeLabel;
import ru.yandex.direct.core.entity.mobilecontent.model.OsType;
import ru.yandex.direct.intapi.entity.launcher.controller.LauncherController;
import ru.yandex.direct.intapi.entity.launcher.model.App;
import ru.yandex.direct.intapi.entity.launcher.model.GeoRestriction;
import ru.yandex.direct.intapi.entity.launcher.model.LauncherCampaignsResponse;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static java.util.Comparator.comparing;
import static java.util.function.BinaryOperator.maxBy;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.banner.service.BannerUtils.getValueIfAssignable;

@Service
public class LauncherService {

    private final CampaignService campaignService;
    private final AdGroupService adGroupService;
    private final BannerService bannerService;

    @Autowired
    public LauncherService(CampaignService campaignService,
                           AdGroupService adGroupService,
                           BannerService bannerService) {
        this.campaignService = campaignService;
        this.adGroupService = adGroupService;
        this.bannerService = bannerService;
    }

    public LauncherCampaignsResponse getCampaigns() {
        return this.getLauncherCampaigns();
    }

    public LauncherCampaignsResponse getLauncherCampaigns() {
        final List<CampaignForLauncher> campaigns = campaignService.getCampaignsForLauncher().stream()
                .filter(c -> c.getStrategy() != null && c.getStrategy().getStrategyData() != null
                        && c.getStrategy().getStrategyData().getAvgCpi() != null)
                .filter(c -> c.getDisabledDomains() == null || !c.getDisabledDomains().contains("com.yandex.launcher"))
                .collect(toList());
        final Map<Long, CampaignForLauncher> campaignsMap =
                campaigns.stream().collect(toMap(CampaignForLauncher::getId, identity()));

        final Set<Long> campaignIds = campaigns.stream()
                .map(CampaignForLauncher::getId).collect(toSet());
        // todo: add osType and statuses to AdGroupsSelectionCriteria
        final List<AdGroup> adGroups = adGroupService
                .getAdGroupsBySelectionCriteria(new AdGroupsSelectionCriteria()
                                .withCampaignIds(campaignIds)
                                .withAdGroupStatuses(AdGroupStatus.ACCEPTED)
                                .withAdGroupTypes(AdGroupType.MOBILE_CONTENT),
                        LimitOffset.maxLimited(),
                        true);
        final List<MobileContentAdGroup> mobileAdGroups = adGroups
                .stream()
                .filter(g -> g.getClass().isAssignableFrom(MobileContentAdGroup.class))
                .map(MobileContentAdGroup.class::cast)
                .filter(g -> g.getStatusModerate() == StatusModerate.YES
                        && g.getStatusBsSynced() == StatusBsSynced.YES
                        && g.getMobileContent() != null
                        && g.getMobileContent().getOsType() == OsType.ANDROID
                        && !StringUtils.isEmpty(g.getMobileContent().getStoreContentId())
                        && !StringUtils.isEmpty(g.getMinimalOperatingSystemVersion()))
                .collect(toList());
        final Map<Long, MobileContentAdGroup> adGroupMap = mobileAdGroups.stream()
                .collect(toMap(MobileContentAdGroup::getId, identity()));

        final Set<Long> adGroupIds = mobileAdGroups.stream().map(AdGroup::getId).collect(toSet());

        final Map<Long, BannerWithSystemFields> adGroupIdToMainBannerMap =
                bannerService.getBannersByAdGroupIds(adGroupIds).values().stream()
                        .map(banners -> banners.stream()
                                .filter(b -> b.getStatusBsSynced() == StatusBsSynced.YES
                                        && b.getBsBannerId() != null
                                        && Boolean.TRUE.equals(b.getStatusActive())
                                        && Boolean.TRUE.equals(b.getStatusShow())
                                        && !Boolean.TRUE.equals(b.getStatusArchived()))
                                .findFirst().orElse(null))
                        .filter(Objects::nonNull)
                        .collect(toMap(BannerWithSystemFields::getAdGroupId, identity()));

        // todo: закомменчено до https://st.yandex-team.ru/DIRECT-76655: [launcher][direct]
        // Добавить дисклеймер для разных языков
        //final Map<Long, String> bannerDisclaimerMap = new HashMap<>();
        //final Set<Long> bannerIds = adGroupIdToMainBannerMap.values().stream().map(Banner::getId).collect(toSet());
        //shardHelper.groupByShard(bannerIds, ShardKey.BID, identity())
        //        .stream()
        //        .mapKeyValue(bannerAdditionsRepository::getAdditionDisclaimerByBannerIds)
        //        .forEach(bannerDisclaimerMap::putAll);

        final Collection<App> apps = adGroupIds.stream()
                .filter(id -> adGroupIdToMainBannerMap.get(id) != null)
                .map(id -> {
                    final MobileContentAdGroup adGroup = adGroupMap.get(id);
                    final CampaignForLauncher campaign = campaignsMap.get(adGroup.getCampaignId());
                    final BannerWithSystemFields banner = adGroupIdToMainBannerMap.get(id);
                    final BigDecimal payout = campaign.getStrategy().getStrategyData().getAvgCpi();
                    final String bannerHref = getValueIfAssignable(banner, BannerWithHref.HREF);

                    final List<Long> geo = adGroup.getGeo() != null
                            ? adGroup.getGeo()
                            : campaign.getGeo() != null
                            ? campaign.getGeo().stream().map(Long::valueOf).collect(toList())
                            : null;
                    return new App()
                            .withBkBannerId(banner.getBsBannerId())
                            .withCampaignId(campaign.getId())
                            .withTitle(campaign.getName())
                            .withCurrency(campaign.getCurrency().name())
                            .withGeoRestrictions(geo != null ? toRestrictions(geo) : null)
                            .withMinOsVersion(adGroup.getMinimalOperatingSystemVersion())
                            .withPackageName(adGroup.getMobileContent().getStoreContentId())
                            // todo: закомменчено до https://st.yandex-team.ru/DIRECT-76655: [launcher][direct]
                            // Добавить дисклеймер для разных языков
                            //.withDisclaimer(bannerDisclaimerMap.get(banner.getId()))
                            .withAgeLabel(Optional.ofNullable(adGroup.getMobileContent().getAgeLabel())
                                    .map(AgeLabel::toSource)
                                    .map(al -> al.getLiteral())
                                    .orElse(null))
                            .withPayout(payout)
                            .withCapping(1000L)
                            .withTrackingUrl(bannerHref);
                })
                .collect(toMap(LauncherController.KEY_EXTRACTOR, identity(), maxBy(comparing(App::getPayout))))
                .values();
        return new LauncherCampaignsResponse().withApps(new ArrayList<>(apps));
    }

    private List<GeoRestriction> toRestrictions(List<Long> geo) {
        return geo.stream()
                .filter(id -> id >= 0)
                .map(id -> new GeoRestriction().withId(id).withExceptions(toExceptions(geo, id)))
                .collect(toList());
    }

    private List<Long> toExceptions(List<Long> geo, Long startId) {
        final int fromIndex = geo.indexOf(startId) + 1;
        final Integer toIndex = geo.stream()
                .skip(fromIndex)
                .filter(id -> id >= 0)
                .findFirst()
                .map(geo::indexOf)
                .orElse(geo.size());
        return fromIndex == toIndex ? null : geo.subList(fromIndex, toIndex).stream().map(Math::abs).collect(toList());
    }

}
