package ru.yandex.direct.intapi.entity.metricslog.controller;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.ws.rs.core.MediaType;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;

import ru.yandex.direct.common.log.service.LogMetricsService;
import ru.yandex.direct.common.log.service.metrics.MetricsAddRequest;
import ru.yandex.direct.common.logging.EventType;
import ru.yandex.direct.common.logging.LoggingConfig;
import ru.yandex.direct.intapi.IntApiException;
import ru.yandex.direct.intapi.entity.metricslog.model.SolomonMetric;
import ru.yandex.direct.intapi.validation.kernel.ValidationResultConversionService;
import ru.yandex.direct.intapi.validation.model.IntapiValidationResult;
import ru.yandex.direct.solomon.SolomonResponseMonitorStatus;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.direct.tvm.AllowServices;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.model.WebSuccessResponse;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.validate.LabelsValidator;

import static java.util.Arrays.asList;
import static ru.yandex.direct.solomon.SolomonExternalSystemMonitorService.INTERPRETED_STATUS_LABEL_NAME;
import static ru.yandex.direct.solomon.SolomonExternalSystemMonitorService.REQS_COUNT_SENSOR_NAME;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_2XX;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_4XX;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_5XX;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_UNPARSABLE;
import static ru.yandex.direct.solomon.SolomonUtils.ENV_LABEL_NAME;
import static ru.yandex.direct.solomon.SolomonUtils.EXTERNAL_SYSTEM_LABEL_NAME;
import static ru.yandex.direct.solomon.SolomonUtils.SUB_SYSTEM_LABEL_NAME;
import static ru.yandex.direct.tvm.TvmService.DIRECT_API_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_API_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.DIRECT_INTAPI_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_INTAPI_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_SCRIPTS_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_SCRIPTS_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_TEST;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxMapSize;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.defect.CommonDefects.absentRequiredField;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

@Controller
@RequestMapping(value = "/metrics", produces = MediaType.APPLICATION_JSON)
@Api(tags = "metrics")
public class MetricsController {
    private static final Logger logger = LoggerFactory.getLogger(MetricsController.class);

    private final LogMetricsService logMetricsService;
    private final ValidationResultConversionService validationResultConversionService;

    @Autowired
    public MetricsController(LogMetricsService logMetricsService,
                             ValidationResultConversionService validationResultConversionService) {
        this.logMetricsService = logMetricsService;
        this.validationResultConversionService = validationResultConversionService;
        initMetrics();
    }

    // Для редких событий заранее создаем метрики, чтобы не терять "первую пришедшую точку"
    private static void initMetrics() {
        List<Labels> reqs = new ArrayList<>();

        // DIRECT-151486: алерт на запись в логброкер
        var yabs = Labels.of(EXTERNAL_SYSTEM_LABEL_NAME, "yabs");
        for (SolomonResponseMonitorStatus status : List.of(STATUS_2XX, STATUS_5XX, STATUS_4XX)) {
            Labels labels = yabs.add(SUB_SYSTEM_LABEL_NAME, "direct-banners-log").add(INTERPRETED_STATUS_LABEL_NAME,
                    status.getName());
            reqs.add(labels);
        }
        for (SolomonResponseMonitorStatus status : List.of(STATUS_2XX, STATUS_5XX, STATUS_UNPARSABLE)) {
            Labels labels = yabs.add(SUB_SYSTEM_LABEL_NAME, "bssoap")
                    .add("method", "UpdateData2")
                    .add(INTERPRETED_STATUS_LABEL_NAME, status.getName());
            reqs.add(labels);
            SolomonUtils.getExternalRateMetric(REQS_COUNT_SENSOR_NAME, labels).add(0);
        }

        // DIRECT-123886: собрать алерт для быстрого обнаружения недоступности API метрики
        var metrikaErr = Labels.of(EXTERNAL_SYSTEM_LABEL_NAME, "metrika",
                INTERPRETED_STATUS_LABEL_NAME, STATUS_5XX.getName());
        reqs.add(metrikaErr.add(SUB_SYSTEM_LABEL_NAME, "internal_api").add("method", "direct/counter_goals"));
        reqs.add(metrikaErr.add(SUB_SYSTEM_LABEL_NAME, "internal_api").add("method", "direct/user_counters_num"));
        reqs.add(metrikaErr.add(SUB_SYSTEM_LABEL_NAME, "audience_internal_api")
                .add("method", "direct/retargeting_conditions_by_uids"));

        reqs.forEach(labels -> SolomonUtils.getExternalRateMetric(REQS_COUNT_SENSOR_NAME, labels).add(0));
    }

    @ApiOperation(
            value = "add",
            httpMethod = "POST",
            nickname = "add",
            notes = "Log metrics data to metrics.log"
    )
    @ApiResponses({
            @ApiResponse(code = 200, message = "Ok", response = WebSuccessResponse.class)
    })
    @RequestMapping(path = "/add", method = RequestMethod.POST)
    @ResponseBody
    public WebResponse add(@RequestBody MetricsAddRequest request) {
        var commonContextMap = request.getCommonContext();
        for (var metrics : request.getMetrics()) {
            if (metrics.getContext() == null) {
                metrics.setContext(new HashMap<>());
            }
            if (commonContextMap != null) {
                metrics.addCommonContext(commonContextMap);
            }
        }
        logMetricsService.saveMetrics(asList(request.getMetrics()));
        return new WebSuccessResponse();
    }

    /**
     * Ручка для отправки в соломон агрегированных метрик, например количества сделанных запросов
     * или отправленных объектов.
     * Не подходит для отправки "текущих значений", таких как: uptime, размер или возраст очереди.
     */
    @ApiOperation(
            value = "solomon",
            httpMethod = "POST",
            notes = "Send metrics to Solomon. Currently supports only RATE type of metrics"
    )
    @ApiResponses({
            @ApiResponse(code = 200, message = "Ok", response = WebSuccessResponse.class),
            @ApiResponse(code = 400, message = "Ok", response = IntapiValidationResult.class)
    })
    @ResponseBody
    @PostMapping(path = "/solomon")
    @LoggingConfig(enabled = EventType.ERRORS, logResponseBody = EventType.NONE)
    @AllowServices(production = {DIRECT_WEB_PROD, DIRECT_INTAPI_PROD, DIRECT_API_PROD, DIRECT_SCRIPTS_PROD},
            testing = {DIRECT_DEVELOPER, DIRECT_WEB_TEST, DIRECT_INTAPI_TEST, DIRECT_API_TEST, DIRECT_SCRIPTS_TEST})
    public WebResponse solomon(@RequestBody @Nonnull List<SolomonMetric> metrics) {
        var validationResult = ListValidationBuilder.of(metrics, Defect.class)
                .check(notNull())
                .checkEachBy(this::validateSolomonMetric)
                .getResult();
        if (validationResult.hasAnyErrors()) {
            throw new IntApiException(HttpStatus.BAD_REQUEST,
                    validationResultConversionService.buildIntapiValidationResult(validationResult));
        }

        for (SolomonMetric metric : metrics) {
            Map<String, String> labelsMap = metric.getLabels();
            labelsMap.remove(ENV_LABEL_NAME, SolomonUtils.getCurrentEnv());
            Labels labels = Labels.of(labelsMap);
            String name = metric.getName();
            MetricType type = metric.getType();

            try {
                if (type == MetricType.RATE) {
                    SolomonUtils.getExternalRateMetric(name, labels).add(metric.getLongValue());
                } else {
                    throw new IllegalArgumentException("Metric type " + type + " is not supported");
                }
            } catch (ClassCastException e) {
                logger.error("Incompatible metric type for " + name + labels, e);
                throw e;
            }
        }

        return new WebSuccessResponse();
    }

    @SuppressWarnings("rawtypes")
    private ValidationResult<SolomonMetric, Defect> validateSolomonMetric(SolomonMetric metric) {
        // в валидации оставлена поддержка COUNTER/*GAUGE метрик, хотя такие типы не используются
        var validLabelKey = fromPredicate(LabelsValidator::isKeyValid, (Defect) CommonDefects.invalidValue());
        var validLabelValue = fromPredicate(LabelsValidator::isValueValid, (Defect) CommonDefects.invalidValue());
        var validType = inSet(Set.of(MetricType.RATE));

        ItemValidationBuilder<SolomonMetric, Defect> vb = ItemValidationBuilder.of(metric, Defect.class)
                .check(notNull());

        vb.item(metric.getName(), SolomonMetric.NAME_FIELD_NAME)
                .check(notNull())
                .check(validLabelValue, When.notNull());

        MetricType type = metric.getType();
        boolean isDoubleType = type == MetricType.DGAUGE;
        boolean isLongType = type == MetricType.IGAUGE
                || type == MetricType.COUNTER
                || type == MetricType.RATE;
        vb.item(type, SolomonMetric.TYPE_FIELD_NAME)
                .check(notNull())
                .check(validType, invalidValue(), When.notNull());

        Map<String, String> labels = metric.getLabels();
        var labelsVb = vb.item(labels, SolomonMetric.LABELS_FIELD_NAME)
                .check(notNull());

        if (labels != null) {
            labelsVb.check(maxMapSize(Labels.MAX_LABELS_COUNT));
            labelsVb.item(labels.get(ENV_LABEL_NAME), "env")
                    .check(notNull(), absentRequiredField());

            for (var entry : labels.entrySet()) {
                labelsVb.item(entry.getKey(), "[" + entry.getKey() + "]")
                        .check(notNull())
                        .check(validLabelKey, When.notNull())
                        .item(entry.getValue(), "value")
                        .check(notNull())
                        .check(validLabelValue, When.notNull());
            }
        }

        vb.item(metric.getLongValue(), SolomonMetric.LONG_VALUE_FIELD_NAME)
                .check(notNull(), When.isTrue(isLongType))
                .check(isNull(), When.isTrue(isDoubleType));

        vb.item(metric.getDoubleValue(), SolomonMetric.DOUBLE_VALUE_FIELD_NAME)
                .check(notNull(), When.isTrue(isDoubleType))
                .check(isNull(), When.isTrue(isLongType));

        return vb.getResult();
    }
}
