package ru.yandex.direct.intapi.entity.metrika.controller;

import java.util.List;

import javax.annotation.Nonnull;
import javax.ws.rs.core.MediaType;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.entity.metrika.model.MetrikaBannersParam;
import ru.yandex.direct.intapi.entity.metrika.model.MetrikaBannersResult;
import ru.yandex.direct.intapi.entity.metrika.model.MetrikaCampaignsResult;
import ru.yandex.direct.intapi.entity.metrika.model.MetrikaCountersResult;
import ru.yandex.direct.intapi.entity.metrika.model.MetrikaDynamicsParam;
import ru.yandex.direct.intapi.entity.metrika.model.MetrikaDynamicsResult;
import ru.yandex.direct.intapi.entity.metrika.model.MetrikaGroupsParam;
import ru.yandex.direct.intapi.entity.metrika.model.MetrikaGroupsResult;
import ru.yandex.direct.intapi.entity.metrika.model.MetrikaPerformanceFiltersParam;
import ru.yandex.direct.intapi.entity.metrika.model.MetrikaPerformanceFiltersResult;
import ru.yandex.direct.intapi.entity.metrika.service.MetrikaBannersService;
import ru.yandex.direct.intapi.entity.metrika.service.MetrikaCampaignsService;
import ru.yandex.direct.intapi.entity.metrika.service.MetrikaCountersService;
import ru.yandex.direct.intapi.entity.metrika.service.MetrikaDynamicsService;
import ru.yandex.direct.intapi.entity.metrika.service.MetrikaGroupsService;
import ru.yandex.direct.intapi.entity.metrika.service.MetrikaPerformanceFiltersService;
import ru.yandex.direct.intapi.entity.metrika.service.MetrikaRetargetingGoalsService;
import ru.yandex.direct.intapi.webapp.semaphore.Semaphore;
import ru.yandex.direct.tvm.AllowServices;

import static ru.yandex.direct.intapi.entity.metrika.controller.MetrikaExportController.LOCK_NAME;
import static ru.yandex.direct.intapi.entity.metrika.controller.MetrikaExportController.MAX_SIMULTANEOUS_REQUESTS;
import static ru.yandex.direct.tvm.TvmService.DIRECT_AUTOTESTS;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.METRIKA_AUDIENCE_CRYPTA_SENDER_PROD;
import static ru.yandex.direct.tvm.TvmService.METRIKA_AUDIENCE_CRYPTA_SENDER_TEST;
import static ru.yandex.direct.tvm.TvmService.METRIKA_CLICKLOGD_PROD;
import static ru.yandex.direct.tvm.TvmService.METRIKA_CLICKLOGD_TEST;
import static ru.yandex.direct.tvm.TvmService.METRIKA_DIRECT_DICT_UPDATER_PROD;
import static ru.yandex.direct.tvm.TvmService.METRIKA_DIRECT_DICT_UPDATER_TEST;
import static ru.yandex.direct.tvm.TvmService.METRIKA_SCHEDULERD_PROD;
import static ru.yandex.direct.tvm.TvmService.METRIKA_SCHEDULERD_TEST;
import static ru.yandex.direct.tvm.TvmService.SUPERMODERATION_PROD;
import static ru.yandex.direct.tvm.TvmService.SUPERMODERATION_TEST;

/**
 * Замена перлового api
 * intapi.direct.yandex.ru/RetargetingGoalsForMetrika - protected/Export/RetargetingGoalsForMetrika.pm
 * intapi.direct.yandex.ru/RetargetingConditionsForMetrika - protected/Export/RetargetingConditionsForMetrika.pm
 * intapi.direct.yandex.ru/jsonrpc/MetrikaLookups?method=metrika_get_retargetings
 * тикет https://st.yandex-team.ru/DIRECT-60956
 * <p>
 * Запрещающая {@link AllowServices} на контроллере - нужна пока мы не сделали TVM обязательным в интерсепторе.
 * Потом нужно будет вывести общие права доступа, записать их в нее, и упразднить аннотации на методах.
 */
@RestController
@RequestMapping("metrika-export")
@Api(value = "Экспорт объектов директа в метрику")
@Semaphore(key = LOCK_NAME, permits = MAX_SIMULTANEOUS_REQUESTS)
@AllowServices(production = {})
public class MetrikaExportController {

    static final String LOCK_NAME = "metrika-lookups";
    static final int MAX_SIMULTANEOUS_REQUESTS = 16;

    private final MetrikaRetargetingGoalsService metrikaRetargetingGoalsService;
    private final MetrikaCampaignsService metrikaCampaignsService;
    private final MetrikaBannersService metrikaBannersService;
    private final MetrikaDynamicsService metrikaDynamicsService;
    private final MetrikaGroupsService metrikaGroupsService;
    private final MetrikaPerformanceFiltersService metrikaPerformanceFiltersService;
    private final MetrikaCountersService metrikaCountersService;

    @Autowired
    @SuppressWarnings("checkstyle:parameternumber")
    public MetrikaExportController(
            MetrikaRetargetingGoalsService metrikaRetargetingGoalsService,
            MetrikaCampaignsService metrikaCampaignsService,
            MetrikaBannersService metrikaBannersService,
            MetrikaDynamicsService metrikaDynamicsService,
            MetrikaGroupsService metrikaGroupsService,
            MetrikaPerformanceFiltersService metrikaPerformanceFiltersService,
            MetrikaCountersService metrikaCountersService
    ) {
        this.metrikaRetargetingGoalsService = metrikaRetargetingGoalsService;
        this.metrikaCampaignsService = metrikaCampaignsService;
        this.metrikaBannersService = metrikaBannersService;
        this.metrikaDynamicsService = metrikaDynamicsService;
        this.metrikaGroupsService = metrikaGroupsService;
        this.metrikaPerformanceFiltersService = metrikaPerformanceFiltersService;
        this.metrikaCountersService = metrikaCountersService;
    }


    @ApiOperation(
            value = "Получение доступных целей по всем шардам.",
            notes = "Если параметры не заданы, возвращаются все цели за один запрос.\n"
                    + "Если переданы параметры divisor и reminder, по возвращаются все цели"
                    + " в которых остаток от деления goal_id / divisor == reminder",
            httpMethod = "GET",
            nickname = "getRetargetingGoals"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params",
                            response = ErrorResponse.class),
                    @ApiResponse(code = 403, message = "permission denied",
                            response = ErrorResponse.class)
            }
    )
    @GetMapping(path = "retargeting-goals",
            produces = MediaType.APPLICATION_JSON
    )
    @AllowServices(production = {METRIKA_AUDIENCE_CRYPTA_SENDER_PROD},
            testing = {DIRECT_DEVELOPER, DIRECT_AUTOTESTS, METRIKA_AUDIENCE_CRYPTA_SENDER_TEST})
    public List<Long> getRetargetingGoals(
            @ApiParam(value = "делитель для значения goal_id", required = true)
            @RequestParam(value = "divisor")
            @Nonnull Long divisor,
            @ApiParam(value = "значение остатка от деления goal_id на divisor", required = true)
            @RequestParam(value = "reminder")
            @Nonnull Long reminder
    ) {
        return metrikaRetargetingGoalsService.getGoals(divisor, reminder);
    }

    @ApiOperation(
            value = "Получение списка кампаний по order_id",
            notes = "В ответе будут только те кампании, которые есть в базе\n",
            httpMethod = "POST",
            nickname = "getCampaigns"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params",
                            response = ErrorResponse.class),
                    @ApiResponse(code = 403, message = "permission denied",
                            response = ErrorResponse.class)
            }
    )
    @PostMapping(path = "campaigns",
            consumes = MediaType.APPLICATION_JSON,
            produces = MediaType.APPLICATION_JSON
    )
    @AllowServices(production = {METRIKA_CLICKLOGD_PROD, METRIKA_CLICKLOGD_TEST},
            testing = {DIRECT_DEVELOPER, DIRECT_AUTOTESTS})
    public List<MetrikaCampaignsResult> getCampaigns(
            @ApiParam(value = "Список order_id", required = true)
            @RequestBody @Nonnull List<Long> orderIds) {
        return metrikaCampaignsService.getCampaigns(orderIds);
    }

    @ApiOperation(
            value = "Получение списка динамических условий по order_id:dyn_cond_id",
            notes = "В ответе будут только те условия, которые есть в базе",
            httpMethod = "POST",
            nickname = "getDynamics"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params",
                            response = ErrorResponse.class),
                    @ApiResponse(code = 403, message = "permission denied",
                            response = ErrorResponse.class)
            }
    )
    @PostMapping(path = "dynamics",
            consumes = MediaType.APPLICATION_JSON,
            produces = MediaType.APPLICATION_JSON
    )
    @AllowServices(production = {METRIKA_SCHEDULERD_PROD, METRIKA_DIRECT_DICT_UPDATER_PROD},
            testing = {METRIKA_SCHEDULERD_TEST, DIRECT_DEVELOPER, DIRECT_AUTOTESTS, METRIKA_DIRECT_DICT_UPDATER_TEST})
    public List<MetrikaDynamicsResult> getDynamics(
            @ApiParam(value = "Список пар order_id:dyn_cond_id", required = true)
            @RequestBody @Nonnull List<MetrikaDynamicsParam> params) {
        return metrikaDynamicsService.getDynamics(params);
    }

    @ApiOperation(
            value = "Получение списка фильтров по  order_id:perf_filter_id",
            notes = "В ответе будут только те фильтры, которые есть в базе",
            httpMethod = "POST",
            nickname = "getPerformanceFilters"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params",
                            response = ErrorResponse.class),
                    @ApiResponse(code = 403, message = "permission denied",
                            response = ErrorResponse.class)
            }

    )
    @PostMapping(path = "performance-filters",
            consumes = MediaType.APPLICATION_JSON,
            produces = MediaType.APPLICATION_JSON
    )
    @AllowServices(production = {METRIKA_SCHEDULERD_PROD, METRIKA_DIRECT_DICT_UPDATER_PROD},
            testing = {METRIKA_SCHEDULERD_TEST, DIRECT_DEVELOPER, DIRECT_AUTOTESTS, METRIKA_DIRECT_DICT_UPDATER_TEST})
    public List<MetrikaPerformanceFiltersResult> getPerformanceFilters(
            @ApiParam(value = "Список пар order_id:perf_filter_id", required = true)
            @RequestBody @Nonnull List<MetrikaPerformanceFiltersParam> params) {
        return metrikaPerformanceFiltersService.getPerformanceFilters(params);
    }

    @ApiOperation(
            value = "Получение списка баннеров по  order_id:banner_id",
            notes = "В ответе будут только те баннеры, которые есть в базе",
            httpMethod = "POST",
            nickname = "getBanners"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params",
                            response = ErrorResponse.class),
                    @ApiResponse(code = 403, message = "permission denied",
                            response = ErrorResponse.class)
            }
    )
    @PostMapping(path = "banners",
            consumes = MediaType.APPLICATION_JSON,
            produces = MediaType.APPLICATION_JSON
    )
    @AllowServices(production = {METRIKA_CLICKLOGD_PROD, METRIKA_CLICKLOGD_TEST, METRIKA_SCHEDULERD_PROD,
            SUPERMODERATION_PROD, METRIKA_DIRECT_DICT_UPDATER_PROD},
            testing = {SUPERMODERATION_TEST, METRIKA_SCHEDULERD_TEST, DIRECT_DEVELOPER, DIRECT_AUTOTESTS,
                    METRIKA_DIRECT_DICT_UPDATER_TEST})
    public List<MetrikaBannersResult> getBanners(
            @ApiParam(value = "Список пар order_id:banner_id", required = true)
            @RequestBody @Nonnull List<MetrikaBannersParam> params) {
        return metrikaBannersService.getBanners(params);
    }

    @ApiOperation(
            value = "Получение списка групп по  order_id:group_id",
            notes = "В ответе будут только те группы, которые есть в базе",
            httpMethod = "POST",
            nickname = "getGroups"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params",
                            response = ErrorResponse.class),
                    @ApiResponse(code = 403, message = "permission denied",
                            response = ErrorResponse.class)
            }
    )
    @PostMapping(path = "groups",
            consumes = MediaType.APPLICATION_JSON,
            produces = MediaType.APPLICATION_JSON
    )
    @AllowServices(production = {METRIKA_CLICKLOGD_PROD, METRIKA_CLICKLOGD_TEST, METRIKA_SCHEDULERD_PROD,
            METRIKA_DIRECT_DICT_UPDATER_PROD},
            testing = {METRIKA_SCHEDULERD_TEST, DIRECT_DEVELOPER, DIRECT_AUTOTESTS, METRIKA_DIRECT_DICT_UPDATER_TEST})
    public List<MetrikaGroupsResult> getGroups(
            @ApiParam(value = "Список пар order_id:group_id", required = true)
            @RequestBody @Nonnull List<MetrikaGroupsParam> params) {
        return metrikaGroupsService.getGroups(params);
    }

    @ApiOperation(
            value = "Получение списка счетчиков метрики, указанных в кампаниях",
            httpMethod = "GET",
            nickname = "getCampCounters"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params",
                            response = ErrorResponse.class),
                    @ApiResponse(code = 403, message = "permission denied",
                            response = ErrorResponse.class)
            }
    )
    @GetMapping(path = "metrika-counters",
            produces = MediaType.APPLICATION_JSON
    )
    @AllowServices(production = {METRIKA_SCHEDULERD_PROD},
            testing = {METRIKA_SCHEDULERD_TEST, DIRECT_DEVELOPER, DIRECT_AUTOTESTS})
    public List<MetrikaCountersResult> getCampCounters(
            @ApiParam(value = "делитель для значения campaign_id", required = true)
            @RequestParam(value = "divisor")
            @Nonnull Long divisor,
            @ApiParam(value = "значение остатка от деления campaign_id на divisor", required = true)
            @RequestParam(value = "reminder")
            @Nonnull Long reminder
    ) {
        return metrikaCountersService.getCounters(divisor, reminder);
    }
}
