package ru.yandex.direct.intapi.entity.metrika.controller;

import javax.ws.rs.core.MediaType;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.entity.metrika.model.objectinfo.AdGroupInfo;
import ru.yandex.direct.intapi.entity.metrika.model.objectinfo.BannerInfo;
import ru.yandex.direct.intapi.entity.metrika.model.objectinfo.CampaignInfo;
import ru.yandex.direct.intapi.entity.metrika.model.objectinfo.CreativeInfo;
import ru.yandex.direct.intapi.entity.metrika.model.objectinfo.MetrikaObjectInfoRequest;
import ru.yandex.direct.intapi.entity.metrika.model.objectinfo.MetrikaObjectInfoResponse;
import ru.yandex.direct.intapi.entity.metrika.model.objectinfo.PerformanceFilterInfo;
import ru.yandex.direct.intapi.entity.metrika.model.objectinfo.RetargetingConditionInfo;
import ru.yandex.direct.intapi.entity.metrika.service.objectinfo.MetrikaAdGroupInfoServiceDelegate;
import ru.yandex.direct.intapi.entity.metrika.service.objectinfo.MetrikaBannerInfoServiceDelegate;
import ru.yandex.direct.intapi.entity.metrika.service.objectinfo.MetrikaCampaignInfoServiceDelegate;
import ru.yandex.direct.intapi.entity.metrika.service.objectinfo.MetrikaCreativeInfoServiceDelegate;
import ru.yandex.direct.intapi.entity.metrika.service.objectinfo.MetrikaObjectInfoService;
import ru.yandex.direct.intapi.entity.metrika.service.objectinfo.MetrikaPerformanceFilterInfoServiceDelegate;
import ru.yandex.direct.intapi.entity.metrika.service.objectinfo.MetrikaRetargetingConditionInfoServiceDelegate;
import ru.yandex.direct.tvm.AllowServices;

import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.METRIKA_DIRECT_DICT_UPDATER_PROD;
import static ru.yandex.direct.tvm.TvmService.METRIKA_DIRECT_DICT_UPDATER_TEST;
import static ru.yandex.direct.tvm.TvmService.METRIKA_SCHEDULERD_PROD;
import static ru.yandex.direct.tvm.TvmService.METRIKA_SCHEDULERD_TEST;

/**
 * Замена перлового api
 * http://intapi.direct.yandex.ru/objectinfo
 * <p>
 * Эта ручка перенесена тоже в object-info:
 * intapi.direct.yandex.ru/RetargetingConditionsForMetrika - protected/Export/RetargetingConditionsForMetrika.pm
 * <p>
 * тикет https://st.yandex-team.ru/DIRECT-60956
 * <p>
 * Запрещающая аннотация {@link AllowServices} на контроллере нужна пока мы не сделали TVM обязательным в интерсепторе.
 * Потом нужно будет вывести общие права доступа, записать их в аннотацию {@link AllowServices} на контроллере и
 * упразднить аннотации на методах.
 */
@RestController
@RequestMapping("metrika-export/object_info")
@Api(value = "Экспорт объектов директа в метрику")
@AllowServices(production = {})
public class MetrikaExportObjectInfoController {

    private final MetrikaObjectInfoService metrikaObjectInfoService;
    private final MetrikaCampaignInfoServiceDelegate campaignInfoServiceDelegate;
    private final MetrikaAdGroupInfoServiceDelegate adGroupInfoServiceDelegate;
    private final MetrikaBannerInfoServiceDelegate bannerInfoServiceDelegate;
    private final MetrikaRetargetingConditionInfoServiceDelegate metrikaRetargetingConditionInfoServiceDelegate;
    private final MetrikaPerformanceFilterInfoServiceDelegate metrikaPerformanceFilterInfoServiceDelegate;
    private final MetrikaCreativeInfoServiceDelegate metrikaCreativeInfoServiceDelegate;

    @Autowired
    public MetrikaExportObjectInfoController(
            MetrikaObjectInfoService metrikaObjectInfoService,
            MetrikaCampaignInfoServiceDelegate campaignInfoServiceDelegate,
            MetrikaAdGroupInfoServiceDelegate adGroupInfoServiceDelegate,
            MetrikaBannerInfoServiceDelegate bannerInfoServiceDelegate,
            MetrikaRetargetingConditionInfoServiceDelegate metrikaRetargetingConditionInfoServiceDelegate,
            MetrikaPerformanceFilterInfoServiceDelegate metrikaPerformanceFilterInfoServiceDelegate,
            MetrikaCreativeInfoServiceDelegate metrikaCreativeInfoServiceDelegate) {
        this.metrikaObjectInfoService = metrikaObjectInfoService;
        this.campaignInfoServiceDelegate = campaignInfoServiceDelegate;
        this.adGroupInfoServiceDelegate = adGroupInfoServiceDelegate;
        this.bannerInfoServiceDelegate = bannerInfoServiceDelegate;
        this.metrikaRetargetingConditionInfoServiceDelegate = metrikaRetargetingConditionInfoServiceDelegate;
        this.metrikaPerformanceFilterInfoServiceDelegate = metrikaPerformanceFilterInfoServiceDelegate;
        this.metrikaCreativeInfoServiceDelegate = metrikaCreativeInfoServiceDelegate;
    }

    @ApiOperation(
            value = "Получение данных по всем кампаниям, "
                    + "изменившимся после некоторого момента времени, "
                    + "определяемого временным токеном вида "
                    + "\"дата-время(ISO_LOCAL_DATE_TIME)/id кампании (cid)\"",
            httpMethod = "GET"
    )
    @ApiResponses(
            @ApiResponse(code = 400, message = "Bad params",
                    response = ErrorResponse.class)
    )
    @GetMapping(path = "campaigns",
            produces = MediaType.APPLICATION_JSON
    )
    @AllowServices(production = {METRIKA_SCHEDULERD_PROD},
            testing = {METRIKA_SCHEDULERD_TEST, DIRECT_DEVELOPER})
    public MetrikaObjectInfoResponse<CampaignInfo> getCampaignsInfo(
            @ApiParam(value = "токен, определяющий, с какого момента запрашиваюстя изменения, "
                    + "получается из ответа на предыдущий запрос "
                    + "(пример: 2017-04-03T10:15:30/1234)", required = false)
            @RequestParam(value = "time_token", required = false) String timeToken,
            @ApiParam(value = "лимит", required = false)
            @RequestParam(value = "limit", required = false) Integer limit
    ) {
        MetrikaObjectInfoRequest request = new MetrikaObjectInfoRequest(timeToken, limit);
        return metrikaObjectInfoService.getObjectsInfo(request, campaignInfoServiceDelegate);
    }


    @ApiOperation(
            value = "Получение данных по всем группам, "
                    + "изменившимся после некоторого момента времени, "
                    + "определяемого временным токеном вида "
                    + "\"дата-время(ISO_LOCAL_DATE_TIME)/id группы (pid)\"",
            httpMethod = "GET"
    )
    @ApiResponses(
            @ApiResponse(code = 400, message = "Bad params",
                    response = ErrorResponse.class)
    )
    @GetMapping(path = "ad_groups",
            produces = MediaType.APPLICATION_JSON
    )
    @AllowServices(production = {METRIKA_SCHEDULERD_PROD, METRIKA_DIRECT_DICT_UPDATER_PROD},
            testing = {METRIKA_SCHEDULERD_TEST, DIRECT_DEVELOPER, METRIKA_DIRECT_DICT_UPDATER_TEST})
    public MetrikaObjectInfoResponse<AdGroupInfo> getAdGroupsInfo(
            @ApiParam(value = "токен, определяющий, с какого момента запрашиваются изменения, "
                    + "получается из ответа на предыдущий запрос "
                    + "(пример: 2017-04-03T10:15:30/1234)", required = false)
            @RequestParam(value = "time_token", required = false) String timeToken,
            @ApiParam(value = "лимит", required = false)
            @RequestParam(value = "limit", required = false) Integer limit
    ) {
        MetrikaObjectInfoRequest request = new MetrikaObjectInfoRequest(timeToken, limit);
        return metrikaObjectInfoService.getObjectsInfo(request, adGroupInfoServiceDelegate);
    }


    @ApiOperation(
            value = "Получение данных по всем баннерам и картиночным баннерам, "
                    + "изменившимся после некоторого момента времени, "
                    + "определяемого временным токеном вида "
                    + "\"дата-время(ISO_LOCAL_DATE_TIME)/(смесь banners.bid и banner_images.image_id - "
                    + "число для сортировки)\"",
            httpMethod = "GET"
    )
    @ApiResponses(
            @ApiResponse(code = 400, message = "Bad params",
                    response = ErrorResponse.class)
    )
    @GetMapping(path = "banners",
            produces = MediaType.APPLICATION_JSON
    )
    @AllowServices(production = {METRIKA_SCHEDULERD_PROD, METRIKA_DIRECT_DICT_UPDATER_PROD},
            testing = {METRIKA_SCHEDULERD_TEST, DIRECT_DEVELOPER, METRIKA_DIRECT_DICT_UPDATER_TEST})
    public MetrikaObjectInfoResponse<BannerInfo> getBannersInfo(
            @ApiParam(value = "токен, определяющий, с какого момента запрашиваются изменения, "
                    + "получается из ответа на предыдущий запрос "
                    + "(пример: 2017-04-03T10:15:30/1234)", required = false)
            @RequestParam(value = "time_token", required = false) String timeToken,
            @ApiParam(value = "лимит", required = false)
            @RequestParam(value = "limit", required = false) Integer limit
    ) {
        MetrikaObjectInfoRequest request = new MetrikaObjectInfoRequest(timeToken, limit);
        return metrikaObjectInfoService.getObjectsInfo(request, bannerInfoServiceDelegate);
    }


    @ApiOperation(
            value = "Получение данных по всем условиям ретаргетинга, "
                    + "изменившимся после некоторого момента времени, "
                    + "определяемого временным токеном вида "
                    + "\"дата-время(ISO_LOCAL_DATE_TIME)/id условия ретаргетинга (ret_cond_id)\"",
            httpMethod = "GET"
    )
    @ApiResponses(
            @ApiResponse(code = 400, message = "Bad params",
                    response = ErrorResponse.class)
    )
    @GetMapping(path = "retargeting_conditions",
            produces = MediaType.APPLICATION_JSON
    )
    @AllowServices(production = {METRIKA_SCHEDULERD_PROD},
            testing = {METRIKA_SCHEDULERD_TEST, DIRECT_DEVELOPER})
    public MetrikaObjectInfoResponse<RetargetingConditionInfo> getConditions(
            @ApiParam(value = "токен, определяющий, с какого момента запрашиваются изменения, "
                    + "получается из ответа на предыдущий запрос "
                    + "(пример: 2017-04-03T10:15:30/1234)", required = false)
            @RequestParam(value = "time_token", required = false) String timeToken,
            @ApiParam(value = "лимит", required = false)
            @RequestParam(value = "limit", required = false) Integer limit
    ) {
        MetrikaObjectInfoRequest request = new MetrikaObjectInfoRequest(timeToken, limit);
        return metrikaObjectInfoService.getObjectsInfo(request, metrikaRetargetingConditionInfoServiceDelegate);
    }


    @ApiOperation(
            value = "Получение данных по всем performance-фильтрам, "
                    + "изменившимся после некоторого момента времени, "
                    + "определяемого временным токеном вида "
                    + "\"дата-время(ISO_LOCAL_DATE_TIME)/id фильтра (perf_filter_id)\"",
            httpMethod = "GET"
    )
    @ApiResponses(
            @ApiResponse(code = 400, message = "Bad params",
                    response = ErrorResponse.class)
    )
    @GetMapping(path = "perf_filters",
            produces = MediaType.APPLICATION_JSON
    )
    @AllowServices(production = {METRIKA_SCHEDULERD_PROD, METRIKA_DIRECT_DICT_UPDATER_PROD},
            testing = {METRIKA_SCHEDULERD_TEST, DIRECT_DEVELOPER, METRIKA_DIRECT_DICT_UPDATER_TEST})
    public MetrikaObjectInfoResponse<PerformanceFilterInfo> getPerfFilters(
            @ApiParam(value = "токен, определяющий, с какого момента запрашиваются изменения, "
                    + "получается из ответа на предыдущий запрос "
                    + "(пример: 2017-04-03T10:15:30/1234)", required = false)
            @RequestParam(value = "time_token", required = false) String timeToken,
            @ApiParam(value = "лимит", required = false)
            @RequestParam(value = "limit", required = false) Integer limit
    ) {
        MetrikaObjectInfoRequest request = new MetrikaObjectInfoRequest(timeToken, limit);
        return metrikaObjectInfoService.getObjectsInfo(request, metrikaPerformanceFilterInfoServiceDelegate);
    }


    @ApiOperation(
            value = "Получение данных по всем креативам, "
                    + "изменившимся после некоторого момента времени, "
                    + "определяемого временным токеном вида "
                    + "\"дата-время(ISO_LOCAL_DATE_TIME)/id креатива (creative_id)\"",
            httpMethod = "GET"
    )
    @ApiResponses(
            @ApiResponse(code = 400, message = "Bad params",
                    response = ErrorResponse.class)
    )
    @GetMapping(path = "creatives",
            produces = MediaType.APPLICATION_JSON
    )
    @AllowServices(production = {METRIKA_SCHEDULERD_PROD},
            testing = {METRIKA_SCHEDULERD_TEST, DIRECT_DEVELOPER})
    public MetrikaObjectInfoResponse<CreativeInfo> getCreatives(
            @ApiParam(value = "токен, определяющий, с какого момента запрашиваются изменения, "
                    + "получается из ответа на предыдущий запрос "
                    + "(пример: 2017-04-03T10:15:30/1234)", required = false)
            @RequestParam(value = "time_token", required = false) String timeToken,
            @ApiParam(value = "лимит", required = false)
            @RequestParam(value = "limit", required = false) Integer limit
    ) {
        MetrikaObjectInfoRequest request = new MetrikaObjectInfoRequest(timeToken, limit);
        return metrikaObjectInfoService.getObjectsInfo(request, metrikaCreativeInfoServiceDelegate);
    }
}
